"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2024 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParseTimeLimitRuleException = exports.TimelimitRule = void 0;
const minuteofday_1 = require("../util/minuteofday");
const token_1 = require("../util/token");
class TimelimitRule {
    constructor({ ruleId, categoryId, maxTimeInMillis, dayMask, applyToExtraTimeUsage, start, end, sessionDurationMilliseconds, sessionPauseMilliseconds, perDay, expiresAt }) {
        this.serialize = () => ({
            ruleId: this.ruleId,
            categoryId: this.categoryId,
            time: this.maxTimeInMillis,
            days: this.dayMask,
            extraTime: this.applyToExtraTimeUsage,
            start: this.start,
            end: this.end,
            pause: this.sessionPauseMilliseconds,
            dur: this.sessionDurationMilliseconds,
            e: this.expiresAt
        });
        this.ruleId = ruleId;
        this.categoryId = categoryId;
        this.maxTimeInMillis = maxTimeInMillis;
        this.dayMask = dayMask;
        this.applyToExtraTimeUsage = applyToExtraTimeUsage;
        this.start = start;
        this.end = end;
        this.sessionDurationMilliseconds = sessionDurationMilliseconds;
        this.sessionPauseMilliseconds = sessionPauseMilliseconds;
        this.perDay = perDay;
        this.expiresAt = expiresAt;
        (0, token_1.assertIdWithinFamily)(ruleId);
        (0, token_1.assertIdWithinFamily)(categoryId);
        if (maxTimeInMillis < 0 || (!Number.isSafeInteger(maxTimeInMillis))) {
            throw new ParseTimeLimitRuleException('maxTimeInMillis must be >= 0');
        }
        if (!(Number.isSafeInteger(dayMask) ||
            dayMask < 0 ||
            dayMask > (1 | 2 | 4 | 8 | 16 | 32 | 64))) {
            throw new ParseTimeLimitRuleException('invalid day mask');
        }
        if ((!Number.isSafeInteger(start)) ||
            (!Number.isSafeInteger(end)) ||
            (!Number.isSafeInteger(sessionDurationMilliseconds)) ||
            (!Number.isSafeInteger(sessionPauseMilliseconds))) {
            throw new ParseTimeLimitRuleException();
        }
        if (start < minuteofday_1.MinuteOfDay.MIN || end > minuteofday_1.MinuteOfDay.MAX || start > end) {
            throw new ParseTimeLimitRuleException();
        }
        if (sessionDurationMilliseconds < 0 || sessionPauseMilliseconds < 0) {
            throw new ParseTimeLimitRuleException();
        }
        if (expiresAt !== undefined) {
            if (!Number.isSafeInteger(expiresAt))
                throw new ParseTimeLimitRuleException();
            if (expiresAt <= 0) {
                throw new ParseTimeLimitRuleException();
            }
        }
    }
}
exports.TimelimitRule = TimelimitRule;
TimelimitRule.parse = ({ ruleId, categoryId, time, days, extraTime, start, end, dur, pause, perDay, e }) => (new TimelimitRule({
    ruleId,
    categoryId,
    maxTimeInMillis: time,
    dayMask: days,
    applyToExtraTimeUsage: extraTime,
    start: start !== null && start !== void 0 ? start : minuteofday_1.MinuteOfDay.MIN,
    end: end !== null && end !== void 0 ? end : minuteofday_1.MinuteOfDay.MAX,
    sessionDurationMilliseconds: dur !== null && dur !== void 0 ? dur : 0,
    sessionPauseMilliseconds: pause !== null && pause !== void 0 ? pause : 0,
    perDay: perDay !== null && perDay !== void 0 ? perDay : false,
    expiresAt: e
}));
class ParseTimeLimitRuleException extends Error {
}
exports.ParseTimeLimitRuleException = ParseTimeLimitRuleException;
//# sourceMappingURL=timelimitrule.js.map