"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const http_1 = require("http");
const process_1 = require("process");
const api_1 = require("./api");
const config_1 = require("./config");
const connected_devices_1 = require("./connected-devices");
const database_1 = require("./database");
const inmemoryeventhandler_1 = require("./monitoring/inmemoryeventhandler");
const websocket_1 = require("./websocket");
const worker_1 = require("./worker");
async function main() {
    await database_1.defaultUmzug.up();
    const database = database_1.defaultDatabase;
    const eventHandler = new inmemoryeventhandler_1.InMemoryEventHandler();
    await (0, database_1.assertNestedTransactionsAreWorking)(database);
    await (0, database_1.assertSerializeableTransactionsAreWorking)(database);
    const connectedDevicesManager = new connected_devices_1.VisibleConnectedDevicesManager({
        database
    });
    const { websocketApi, websocketServer } = (0, websocket_1.createWebsocketHandler)({
        database,
        connectedDevicesManager
    });
    (0, worker_1.initWorkers)({
        websocket: websocketApi,
        database
    });
    const api = (0, api_1.createApi)({
        database,
        websocket: websocketApi,
        connectedDevicesManager,
        eventHandler
    });
    const server = new http_1.Server(api);
    websocketServer.attach(server, {
        serveClient: false,
        pingInterval: config_1.config.pingInterval
    });
    const port = process.env.PORT || 8080;
    if (port === 'socketactivation') {
        if (process.env.LISTEN_FDS !== '1') {
            console.warn('expecting exactly one file descriptor for the socket activation');
            process.exit(1);
        }
        else if (process.env.LISTEN_PID !== process_1.pid.toString(10)) {
            console.warn('expecting handover of file descriptors to this process for the socket activation');
            process.exit(1);
        }
        // the sockets are passed using fd 3 + index (with index = 0 in this case)
        server.listen({ fd: 3 });
    }
    else {
        server.listen(port);
    }
    console.log('ready');
}
main().catch((ex) => {
    console.warn(ex);
    process.exit(1);
});
//# sourceMappingURL=index.js.map