"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.dispatchSetParentCategory = dispatchSetParentCategory;
const category_1 = require("../../../../util/category");
const index_1 = require("../exception/index");
const missing_item_1 = require("../exception/missing-item");
const self_limit_1 = require("../exception/self-limit");
async function dispatchSetParentCategory({ action, cache, fromChildSelfLimitAddChildUserId }) {
    const categoryEntry = await cache.database.category.findOne({
        where: {
            familyId: cache.familyId,
            categoryId: action.categoryId
        },
        transaction: cache.transaction
    });
    if (!categoryEntry) {
        throw new missing_item_1.MissingCategoryException();
    }
    if (fromChildSelfLimitAddChildUserId !== null) {
        if (categoryEntry.childId !== fromChildSelfLimitAddChildUserId) {
            throw new self_limit_1.CanNotModifyOtherUsersBySelfLimitationException();
        }
    }
    if (action.parentCategory !== '') {
        const categoriesByUserId = (await cache.database.category.findAll({
            where: {
                familyId: cache.familyId,
                childId: categoryEntry.childId
            },
            attributes: ['categoryId', 'parentCategoryId'],
            transaction: cache.transaction
        })).map((item) => ({
            categoryId: item.categoryId,
            parentCategoryId: item.parentCategoryId
        }));
        if (categoriesByUserId.find((item) => item.categoryId === action.parentCategory) === undefined) {
            throw new missing_item_1.MissingCategoryException();
        }
        const childCategoryIds = new Set();
        {
            const processedCategoryIds = new Set();
            const handle = (currentCategoryId) => {
                if (processedCategoryIds.has(currentCategoryId))
                    return;
                processedCategoryIds.add(currentCategoryId);
                const childCategories = categoriesByUserId.filter((item) => item.parentCategoryId === currentCategoryId);
                childCategories.forEach((childCategory) => {
                    childCategoryIds.add(childCategory.categoryId);
                    handle(childCategory.categoryId);
                });
            };
            handle(action.categoryId);
        }
        if (childCategoryIds.has(action.parentCategory) || action.parentCategory === action.categoryId) {
            throw new index_1.ApplyActionException({
                staticMessage: 'can not set a category as parent which is a child of the category'
            });
        }
        if (fromChildSelfLimitAddChildUserId !== null) {
            try {
                const ownParentCategory = categoriesByUserId.find((item) => item.categoryId === categoryEntry.parentCategoryId);
                const enableDueToLimitAddingWhenChild = ownParentCategory === undefined ||
                    (0, category_1.getCategoryWithParentCategories)(categoriesByUserId, action.parentCategory).indexOf(ownParentCategory.categoryId) !== -1;
                if (!enableDueToLimitAddingWhenChild) {
                    throw new self_limit_1.SelfLimitationException({
                        staticMessage: 'can not change parent categories in a way which reduces limits'
                    });
                }
            }
            catch (ex) {
                if (ex instanceof category_1.GetParentCategoriesException) {
                    throw new missing_item_1.MissingCategoryException();
                }
                else
                    throw ex;
            }
        }
    }
    await cache.database.category.update({
        parentCategoryId: action.parentCategory
    }, {
        where: {
            familyId: cache.familyId,
            categoryId: action.categoryId
        },
        transaction: cache.transaction
    });
    cache.categoriesWithModifiedBaseData.add(action.categoryId);
    cache.incrementTriggeredSyncLevel(2);
}
//# sourceMappingURL=setparentcategory.js.map