"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.dispatchReviewChildTaskAction = dispatchReviewChildTaskAction;
const illegal_state_1 = require("../exception/illegal-state");
const missing_item_1 = require("../exception/missing-item");
const premium_1 = require("../exception/premium");
async function dispatchReviewChildTaskAction({ action, cache }) {
    if (action.ok && cache.hasFullVersion === false) {
        throw new premium_1.PremiumVersionMissingException();
    }
    const taskInfo = await cache.database.childTask.findOne({
        where: {
            familyId: cache.familyId,
            taskId: action.taskId
        },
        transaction: cache.transaction
    });
    if (taskInfo === null)
        throw new missing_item_1.MissingTaskException();
    if (taskInfo.pendingRequest === 0)
        throw new illegal_state_1.IllegalStateException({ staticMessage: 'no task review pending' });
    if (action.ok) {
        const categoryInfoUnsafe = await cache.database.category.findOne({
            where: {
                familyId: cache.familyId,
                categoryId: taskInfo.categoryId
            },
            attributes: ['extraTimeInMillis', 'extraTimeDay'],
            transaction: cache.transaction
        });
        if (categoryInfoUnsafe === null) {
            throw new illegal_state_1.IllegalStateException({ staticMessage: 'category referenced from task not found' });
        }
        const categoryInfo = {
            extraTimeInMillis: categoryInfoUnsafe.extraTimeInMillis,
            extraTimeDay: categoryInfoUnsafe.extraTimeDay
        };
        const resetDayBoundExtraTime = categoryInfo.extraTimeDay !== -1 &&
            action.day !== undefined && categoryInfo.extraTimeDay !== action.day;
        if (resetDayBoundExtraTime) {
            await cache.database.category.update({
                extraTimeInMillis: taskInfo.extraTimeDuration,
                extraTimeDay: -1
            }, {
                where: {
                    familyId: cache.familyId,
                    categoryId: taskInfo.categoryId
                },
                transaction: cache.transaction
            });
        }
        else {
            await cache.database.category.update({
                extraTimeInMillis: categoryInfo.extraTimeInMillis + taskInfo.extraTimeDuration
            }, {
                where: {
                    familyId: cache.familyId,
                    categoryId: taskInfo.categoryId
                },
                transaction: cache.transaction
            });
        }
        cache.categoriesWithModifiedBaseData.add(taskInfo.categoryId);
        await cache.database.childTask.update({
            pendingRequest: 0,
            lastGrantTimestamp: action.time.toString(10)
        }, {
            where: {
                familyId: cache.familyId,
                taskId: action.taskId
            },
            transaction: cache.transaction
        });
        cache.incrementTriggeredSyncLevel(2);
    }
    else {
        await cache.database.childTask.update({
            pendingRequest: 0
        }, {
            where: {
                familyId: cache.familyId,
                taskId: action.taskId
            },
            transaction: cache.transaction
        });
    }
    cache.categoriesWithModifiedTasks.add(taskInfo.categoryId);
    cache.incrementTriggeredSyncLevel(1);
}
//# sourceMappingURL=reviewchildtaskaction.js.map