"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.dispatchParentAction = dispatchParentAction;
const serialization_1 = require("../../../../action/serialization");
const validator_1 = require("../../../../api/validator");
const userflags_1 = require("../../../../model/userflags");
const dispatch_parent_action_1 = require("../dispatch-parent-action");
const illegal_state_1 = require("../exception/illegal-state");
const self_limit_1 = require("../exception/self-limit");
const helper_1 = require("./helper");
async function dispatchParentAction({ action, eventHandler, cache, isChildLimitAdding, deviceId, authentication }) {
    return (0, helper_1.dispatch)({
        action,
        eventHandler,
        type: 'parent',
        validator: validator_1.isSerializedParentAction,
        parser: serialization_1.parseParentAction,
        applier: async (parsedAction) => {
            if (isChildLimitAdding) {
                const deviceEntryUnsafe = await cache.database.device.findOne({
                    attributes: ['currentUserId'],
                    where: {
                        familyId: cache.familyId,
                        deviceId,
                        currentUserId: action.userId
                    },
                    transaction: cache.transaction
                });
                if (!deviceEntryUnsafe) {
                    throw new illegal_state_1.SourceDeviceNotFoundException();
                }
                const deviceUserId = deviceEntryUnsafe.currentUserId;
                if (!deviceUserId) {
                    throw new self_limit_1.SelfLimitNotPossibleException({
                        staticMessage: 'no device user id set but child add self limit action requested'
                    });
                }
                const deviceUserEntryUnsafe = await cache.database.user.findOne({
                    attributes: ['flags'],
                    where: {
                        familyId: cache.familyId,
                        userId: deviceUserId,
                        type: 'child'
                    },
                    transaction: cache.transaction
                });
                if (!deviceUserEntryUnsafe) {
                    throw new self_limit_1.SelfLimitNotPossibleException({
                        staticMessage: 'no child user found for child limit adding action'
                    });
                }
                if ((parseInt(deviceUserEntryUnsafe.flags, 10) & userflags_1.UserFlags.ALLOW_SELF_LIMIT_ADD) !== userflags_1.UserFlags.ALLOW_SELF_LIMIT_ADD) {
                    throw new self_limit_1.SelfLimitNotPossibleException({
                        staticMessage: 'child add limit action found but not allowed'
                    });
                }
                await (0, dispatch_parent_action_1.dispatchParentAction)({
                    action: parsedAction,
                    cache,
                    parentUserId: action.userId,
                    sourceDeviceId: deviceId,
                    fromChildSelfLimitAddChildUserId: deviceUserId,
                    authentication
                });
            }
            else {
                await (0, dispatch_parent_action_1.dispatchParentAction)({
                    action: parsedAction,
                    cache,
                    parentUserId: action.userId,
                    sourceDeviceId: deviceId,
                    fromChildSelfLimitAddChildUserId: null,
                    authentication
                });
            }
        }
    });
}
//# sourceMappingURL=parent-action.js.map