"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.linkMailAddress = void 0;
const http_errors_1 = require("http-errors");
const token_1 = require("../../util/token");
const authentication_1 = require("../authentication");
const websocket_1 = require("../websocket");
const linkMailAddress = async ({ mailAuthToken, deviceAuthToken, parentUserId, parentPasswordSecondHash, database, websocket }) => {
    await database.transaction(async (transaction) => {
        const deviceEntry = await database.device.findOne({
            where: {
                deviceAuthToken
            },
            transaction
        });
        if (!deviceEntry) {
            throw new http_errors_1.Unauthorized();
        }
        const familyId = deviceEntry.familyId;
        const mailInfo = await (0, authentication_1.requireMailAndLocaleByAuthToken)({ mailAuthToken, database, transaction, invalidate: true });
        const exisitingUser = await database.user.findOne({
            where: {
                mail: mailInfo.mail
            },
            transaction
        });
        if (exisitingUser) {
            throw new http_errors_1.Conflict();
        }
        const parentEntry = await database.user.findOne({
            where: {
                type: 'parent',
                familyId,
                userId: parentUserId
            },
            transaction
        });
        if (!parentEntry) {
            throw new http_errors_1.Conflict();
        }
        if (parentEntry.mail !== '') {
            throw new http_errors_1.Conflict();
        }
        if (parentEntry.secondPasswordHash !== parentPasswordSecondHash) {
            throw new http_errors_1.Conflict();
        }
        if (!parentEntry.secondPasswordSalt) {
            throw new http_errors_1.Conflict();
        }
        parentEntry.mail = mailInfo.mail;
        await parentEntry.save({ transaction });
        // invalidiate client caches
        await database.family.update({
            userListVersion: (0, token_1.generateVersionId)()
        }, {
            where: {
                familyId
            },
            transaction
        });
        // notify
        await (0, websocket_1.notifyClientsAboutChangesDelayed)({
            familyId,
            sourceDeviceId: null,
            generalLevel: 1,
            targetedLevels: new Map(),
            database,
            websocket,
            transaction
        });
    });
};
exports.linkMailAddress = linkMailAddress;
//# sourceMappingURL=link-mail-address.js.map