"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SharedSecretException = void 0;
exports.getSharedSecret = getSharedSecret;
const crypto_1 = require("crypto");
const devicedhkey_1 = require("../../database/devicedhkey");
const token_1 = require("../../util/token");
async function getSharedSecret({ database, transaction, familyId, deviceId, keyId, otherPublicKey }) {
    if (!(0, token_1.isVersionId)(keyId))
        throw new KeyNotFoundException('invalid key id');
    const databaseKeyEntry = await database.deviceDhKey.findOne({
        where: {
            familyId,
            deviceId,
            version: keyId
        },
        transaction
    });
    if (!databaseKeyEntry)
        throw new KeyNotFoundException('private key not found');
    if (databaseKeyEntry.expireAt === null) {
        databaseKeyEntry.expireAt = (0, devicedhkey_1.calculateExpireTime)(BigInt(Date.now())).toString(10);
        await databaseKeyEntry.save({ transaction });
    }
    else {
        if (BigInt(databaseKeyEntry.expireAt) < BigInt(Date.now()))
            throw new KeyExpiredException();
    }
    const privateKey = (() => {
        try {
            return (0, crypto_1.createPrivateKey)({
                key: databaseKeyEntry.privateKey,
                format: 'der',
                type: 'pkcs8'
            });
        }
        catch (ex) {
            throw new MalformedPrivateKeyException();
        }
    })();
    const decodedOtherPublicKey = (() => {
        try {
            return (0, crypto_1.createPublicKey)({
                key: otherPublicKey,
                format: 'der',
                type: 'spki'
            });
        }
        catch (ex) {
            throw new MalformedPublicKeyException();
        }
    })();
    const sharedSecret = (() => {
        try {
            return (0, crypto_1.diffieHellman)({
                privateKey,
                publicKey: decodedOtherPublicKey
            });
        }
        catch (ex) {
            throw new MalformedNoMatchingKeysException();
        }
    })();
    return {
        sharedSecret,
        ownPublicKey: databaseKeyEntry.publicKey
    };
}
class SharedSecretException extends Error {
}
exports.SharedSecretException = SharedSecretException;
class MalformedPrivateKeyException extends SharedSecretException {
    constructor() { super('private key'); }
}
class MalformedPublicKeyException extends SharedSecretException {
    constructor() { super('public key'); }
}
class MalformedNoMatchingKeysException extends SharedSecretException {
    constructor() { super('no matching keys'); }
}
class KeyExpiredException extends SharedSecretException {
    constructor() { super('key expired'); }
}
class KeyNotFoundException extends SharedSecretException {
    constructor(message) { super('key not found: ' + message); }
}
//# sourceMappingURL=shared-secret.js.map