"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setPrimaryDevice = void 0;
const http_errors_1 = require("http-errors");
const config_1 = require("../../config");
const token_1 = require("../../util/token");
const websocket_1 = require("../websocket");
const setPrimaryDevice = async ({ database, websocket, deviceAuthToken, currentUserId, action }) => {
    return database.transaction(async (transaction) => {
        const deviceEntryUnsafe = await database.device.findOne({
            where: {
                deviceAuthToken
            },
            transaction,
            attributes: ['familyId', 'currentUserId', 'deviceId']
        });
        if (!deviceEntryUnsafe) {
            throw new http_errors_1.Unauthorized();
        }
        const deviceEntry = {
            familyId: deviceEntryUnsafe.familyId,
            currentUserId: deviceEntryUnsafe.currentUserId,
            deviceId: deviceEntryUnsafe.deviceId
        };
        if ((deviceEntry.currentUserId !== currentUserId) || (currentUserId === '')) {
            throw new http_errors_1.Conflict();
        }
        const userEntryUnsafe = await database.user.findOne({
            where: {
                familyId: deviceEntry.familyId,
                userId: deviceEntry.currentUserId
            },
            transaction,
            attributes: ['currentDevice']
        });
        if (!userEntryUnsafe) {
            throw new http_errors_1.Conflict();
        }
        const userEntry = {
            currentDevice: userEntryUnsafe.currentDevice
        };
        const userDeviceEntriesUnsafe = await database.device.findAll({
            where: {
                familyId: deviceEntry.familyId,
                currentUserId
            },
            transaction,
            attributes: ['deviceId']
        });
        const userDeviceEntries = userDeviceEntriesUnsafe.map((item) => ({
            deviceId: item.deviceId
        }));
        if (userDeviceEntries.length >= 2) {
            const familyEntryUnsafe = await database.family.findOne({
                where: {
                    familyId: deviceEntry.familyId
                },
                transaction,
                attributes: ['hasFullVersion']
            });
            if (!familyEntryUnsafe) {
                throw new http_errors_1.Conflict();
            }
            const familyEntry = {
                hasFullVersion: familyEntryUnsafe.hasFullVersion
            };
            if (!(familyEntry.hasFullVersion || config_1.config.alwaysPro)) {
                return 'requires full version';
            }
        }
        if (action === 'set this device') {
            // check that no other device is selected
            if (userDeviceEntries.find((item) => item.deviceId === userEntry.currentDevice)) {
                return 'assigned to other device';
            }
            // update
            const [affectedRows] = await database.user.update({
                currentDevice: deviceEntry.deviceId
            }, {
                where: {
                    familyId: deviceEntry.familyId,
                    userId: currentUserId
                },
                transaction
            });
            if (affectedRows !== 1) {
                throw new http_errors_1.Conflict();
            }
        }
        else if (action === 'unset this device') {
            if (userEntry.currentDevice !== deviceEntry.deviceId) {
                throw new http_errors_1.Conflict();
            }
            // update
            const [affectedRows] = await database.user.update({
                currentDevice: ''
            }, {
                where: {
                    familyId: deviceEntry.familyId,
                    userId: currentUserId
                },
                transaction
            });
            if (affectedRows !== 1) {
                throw new http_errors_1.Conflict();
            }
        }
        else {
            throw new http_errors_1.InternalServerError('illegal state');
        }
        // invalidiate user list
        await database.family.update({
            userListVersion: (0, token_1.generateVersionId)()
        }, {
            transaction,
            where: {
                familyId: deviceEntry.familyId
            }
        });
        // trigger sync
        await (0, websocket_1.notifyClientsAboutChangesDelayed)({
            familyId: deviceEntry.familyId,
            sourceDeviceId: deviceEntry.deviceId,
            websocket,
            database,
            generalLevel: 1, // the source device knows it already
            targetedLevels: new Map(),
            transaction
        });
        return 'success';
    });
};
exports.setPrimaryDevice = setPrimaryDevice;
//# sourceMappingURL=set-primary-device.js.map