"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SendKeyRequestAction = void 0;
const basetypes_1 = require("./basetypes");
const exception_1 = require("./meta/exception");
const util_1 = require("./meta/util");
const keyrequest_1 = require("../database/keyrequest");
const actionType = 'SendKeyRequestAction';
class SendKeyRequestAction extends basetypes_1.AppLogicAction {
    constructor({ deviceSequenceNumber, deviceId, categoryId, type, tempKey, signature }) {
        super();
        (0, util_1.assertSafeInteger)({ value: deviceSequenceNumber, field: 'deviceSequenceNumber', actionType });
        (0, util_1.assertSafeInteger)({ value: type, field: 'deviceSequenceNumber', actionType });
        if (tempKey.length != 32 || signature.length != 64) {
            throw new exception_1.InvalidActionParameterException({
                actionType,
                staticMessage: 'key/signature has wrong length'
            });
        }
        if (deviceId !== undefined) {
            (0, util_1.assertIdWithinFamily)({ value: deviceId, actionType, field: 'deviceId' });
        }
        if (categoryId !== undefined) {
            (0, util_1.assertIdWithinFamily)({ value: categoryId, actionType, field: 'categoryId' });
        }
        if (deviceId !== undefined && categoryId !== undefined) {
            throw new exception_1.InvalidActionParameterException({
                actionType,
                staticMessage: 'can not specify device and category at the same time'
            });
        }
        if (keyrequest_1.types.all.indexOf(type) === -1) {
            throw new exception_1.InvalidActionParameterException({
                actionType,
                staticMessage: 'invalid type'
            });
        }
        this.deviceSequenceNumber = deviceSequenceNumber;
        this.deviceId = deviceId;
        this.categoryId = categoryId;
        this.type = type;
        this.tempKey = tempKey;
        this.signature = signature;
    }
}
exports.SendKeyRequestAction = SendKeyRequestAction;
SendKeyRequestAction.parse = ({ dsn, deviceId, categoryId, dataType, tempKey, signature }) => (new SendKeyRequestAction({
    deviceSequenceNumber: dsn,
    deviceId,
    categoryId,
    type: dataType,
    tempKey: Buffer.from(tempKey, 'base64'),
    signature: Buffer.from(signature, 'base64')
}));
//# sourceMappingURL=sendkeyrequest.js.map