"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateServerDataStatus = void 0;
const config_1 = require("../../../config");
const statusmessage_1 = require("../../../function/statusmessage");
const category_1 = require("./category");
const device_detail_1 = require("./device-detail");
const device_list_1 = require("./device-list");
const dh_keys_1 = require("./dh-keys");
const family_entry_1 = require("./family-entry");
const user_list_1 = require("./user-list");
const key_requests_1 = require("./key-requests");
const key_responses_1 = require("./key-responses");
const u2f_1 = require("./u2f");
const generateServerDataStatus = async ({ database, clientStatus, familyId, deviceId, transaction, eventHandler }) => {
    const clientLevel = clientStatus.clientLevel || 0;
    const familyEntry = await (0, family_entry_1.getFamilyEntry)({ database, familyId, transaction });
    const doesClientSupportTasks = clientLevel >= 3;
    const doesClientSupportCryptoApps = clientLevel >= 4;
    const doesClientSupportDh = clientLevel >= 5;
    const doesClientSupportU2f = clientLevel >= 6;
    const result = {
        fullVersion: config_1.config.alwaysPro ? 1 : (familyEntry.hasFullVersion ? parseInt(familyEntry.fullVersionUntil, 10) : 0),
        message: await (0, statusmessage_1.getStatusMessage)({ database, transaction }) || undefined,
        apiLevel: 7
    };
    if (familyEntry.deviceListVersion !== clientStatus.devices) {
        result.devices = await (0, device_list_1.getDeviceList)({ database, transaction, familyEntry });
    }
    if (familyEntry.userListVersion !== clientStatus.users) {
        result.users = await (0, user_list_1.getUserList)({ database, transaction, familyEntry });
    }
    const categoryDataToSync = await (0, category_1.getCategoryDataToSync)({ database, transaction, familyEntry, categoriesStatus: clientStatus.categories });
    if (categoryDataToSync.removedCategoryIds.length > 0) {
        result.rmCategories = categoryDataToSync.removedCategoryIds;
    }
    if (categoryDataToSync.categoryIdsToSyncBaseData.length > 0) {
        result.categoryBase = await (0, category_1.getCategoryBaseDatas)({
            database, transaction, familyEntry,
            categoryIdsToSyncBaseData: categoryDataToSync.categoryIdsToSyncBaseData
        });
    }
    if (categoryDataToSync.categoryIdsToSyncAssignedApps.length > 0) {
        result.categoryApp = await (0, category_1.getCategoryAssignedApps)({
            database, transaction, familyEntry,
            serverCategoriesVersions: categoryDataToSync.serverCategoriesVersions,
            categoryIdsToSyncAssignedApps: categoryDataToSync.categoryIdsToSyncAssignedApps
        });
    }
    if (categoryDataToSync.categoryIdsToSyncRules.length > 0) {
        result.rules = await (0, category_1.getRules)({
            database, transaction, familyEntry,
            serverCategoriesVersions: categoryDataToSync.serverCategoriesVersions,
            categoryIdsToSyncRules: categoryDataToSync.categoryIdsToSyncRules
        });
    }
    if (categoryDataToSync.categoryIdsToSyncUsedTimes.length > 0) {
        result.usedTimes = await (0, category_1.getUsedTimes)({
            database, transaction, familyEntry,
            serverCategoriesVersions: categoryDataToSync.serverCategoriesVersions,
            categoryIdsToSyncUsedTimes: categoryDataToSync.categoryIdsToSyncUsedTimes,
            clientLevel: clientStatus.clientLevel || null
        });
    }
    if (categoryDataToSync.categoryIdsToSyncTasks.length > 0 && doesClientSupportTasks) {
        result.tasks = await (0, category_1.getTasks)({
            database, transaction, familyEntry,
            serverCategoriesVersions: categoryDataToSync.serverCategoriesVersions,
            categoryIdsToSyncTasks: categoryDataToSync.categoryIdsToSyncTasks
        });
    }
    if (doesClientSupportCryptoApps) {
        result.devices2 = await (0, device_detail_1.getDeviceDetailList)({
            database,
            transaction,
            familyEntry,
            devicesDetail: clientStatus.devicesDetail || {}
        }) || undefined;
        result.krq = await (0, key_requests_1.getKeyRequests)({
            database,
            transaction,
            familyEntry,
            deviceId,
            lastSeenRequestIndex: clientStatus.kri || null
        }) || undefined;
        result.kr = await (0, key_responses_1.getKeyResponses)({
            database,
            transaction,
            familyEntry,
            deviceId,
            lastSeenRequestIndex: clientStatus.kr || null
        }) || undefined;
    }
    if (doesClientSupportDh) {
        result.dh = await (0, dh_keys_1.getDeviceDhKeys)({
            database,
            transaction,
            familyEntry,
            deviceId,
            lastVersionId: clientStatus.dh || null,
            eventHandler
        }) || undefined;
    }
    if (doesClientSupportU2f) {
        result.u2f = await (0, u2f_1.getU2f)({
            database,
            transaction,
            familyEntry,
            lastVersionId: clientStatus.u2f || null
        }) || undefined;
    }
    return result;
};
exports.generateServerDataStatus = generateServerDataStatus;
//# sourceMappingURL=index.js.map