"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2023 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.dispatchUpdateDeviceStatus = dispatchUpdateDeviceStatus;
const device_1 = require("../../../../database/device");
const newpermissionstatus_1 = require("../../../../model/newpermissionstatus");
const protectionlevel_1 = require("../../../../model/protectionlevel");
const runtimepermissionstatus_1 = require("../../../../model/runtimepermissionstatus");
const enum_1 = require("../../../../util/enum");
const manipulation_1 = require("../../../warningmail/manipulation");
const illegal_state_1 = require("../exception/illegal-state");
async function dispatchUpdateDeviceStatus({ deviceId, action, cache }) {
    const deviceEntry = await cache.database.device.findOne({
        where: {
            familyId: cache.familyId,
            deviceId
        },
        transaction: cache.transaction
    });
    if (!deviceEntry) {
        throw new illegal_state_1.SourceDeviceNotFoundException();
    }
    const hadManipulationBefore = (0, device_1.hasDeviceManipulation)(deviceEntry);
    if (action.newProtetionLevel) {
        const hasChanged = deviceEntry.currentProtectionLevel !== action.newProtetionLevel;
        deviceEntry.currentProtectionLevel = action.newProtetionLevel;
        deviceEntry.highestProtectionLevel = (0, enum_1.enumMax)(deviceEntry.currentProtectionLevel, deviceEntry.highestProtectionLevel, protectionlevel_1.protetionLevels);
        if (hasChanged && (deviceEntry.currentProtectionLevel !== deviceEntry.highestProtectionLevel)) {
            deviceEntry.hadManipulation = true;
            deviceEntry.hadManipulationFlags |= device_1.DeviceHadManipulationFlags.ProtectionLevel;
        }
    }
    if (action.newUsageStatsPermissionStatus) {
        const hasChanged = deviceEntry.currentUsageStatsPermission !== action.newUsageStatsPermissionStatus;
        deviceEntry.currentUsageStatsPermission = action.newUsageStatsPermissionStatus;
        deviceEntry.highestUsageStatsPermission = (0, enum_1.enumMax)(deviceEntry.currentUsageStatsPermission, deviceEntry.highestUsageStatsPermission, runtimepermissionstatus_1.runtimePermissionStatusValues);
        if (hasChanged && (deviceEntry.currentUsageStatsPermission !== deviceEntry.highestUsageStatsPermission)) {
            deviceEntry.hadManipulation = true;
            deviceEntry.hadManipulationFlags |= device_1.DeviceHadManipulationFlags.UsageStatsAccess;
        }
    }
    if (action.newNotificationAccessPermission) {
        const hasChanged = deviceEntry.currentNotificationAccessPermission !== action.newNotificationAccessPermission;
        deviceEntry.currentNotificationAccessPermission = action.newNotificationAccessPermission;
        deviceEntry.highestNotificationAccessPermission = (0, enum_1.enumMax)(deviceEntry.currentNotificationAccessPermission, deviceEntry.highestNotificationAccessPermission, newpermissionstatus_1.newPermissionStatusValues);
        if (hasChanged && (deviceEntry.currentNotificationAccessPermission !== deviceEntry.highestNotificationAccessPermission)) {
            deviceEntry.hadManipulation = true;
            deviceEntry.hadManipulationFlags |= device_1.DeviceHadManipulationFlags.NotificationAccess;
        }
    }
    if (action.newOverlayPermission) {
        const hasChanged = deviceEntry.currentOverlayPermission !== action.newOverlayPermission;
        deviceEntry.currentOverlayPermission = action.newOverlayPermission;
        deviceEntry.highestOverlayPermission = (0, enum_1.enumMax)(deviceEntry.currentOverlayPermission, deviceEntry.highestOverlayPermission, runtimepermissionstatus_1.runtimePermissionStatusValues);
        if (hasChanged && (deviceEntry.currentOverlayPermission !== deviceEntry.highestOverlayPermission)) {
            deviceEntry.hadManipulation = true;
            deviceEntry.hadManipulationFlags |= device_1.DeviceHadManipulationFlags.OverlayPermission;
        }
    }
    if (action.newAccessibilityServiceEnabled !== undefined) {
        const hasChanged = deviceEntry.asEnabled !== action.newAccessibilityServiceEnabled;
        deviceEntry.asEnabled = action.newAccessibilityServiceEnabled;
        if (action.newAccessibilityServiceEnabled) {
            deviceEntry.wasAsEnabled = true;
        }
        if (hasChanged && (deviceEntry.asEnabled !== deviceEntry.wasAsEnabled)) {
            deviceEntry.hadManipulation = true;
            deviceEntry.hadManipulationFlags |= device_1.DeviceHadManipulationFlags.AccessibiityService;
        }
    }
    if (action.newAppVersion !== undefined) {
        const hasChanged = deviceEntry.currentAppVersion !== action.newAppVersion;
        deviceEntry.currentAppVersion = action.newAppVersion;
        deviceEntry.highestAppVersion = Math.max(deviceEntry.currentAppVersion, deviceEntry.highestAppVersion);
        if (hasChanged && (deviceEntry.currentAppVersion !== deviceEntry.highestAppVersion)) {
            deviceEntry.hadManipulation = true;
            deviceEntry.hadManipulationFlags |= device_1.DeviceHadManipulationFlags.AppVersion;
        }
    }
    if (action.didReboot) {
        deviceEntry.didReboot = true;
    }
    if (action.isQOrLaterNow) {
        const hasChanged = deviceEntry.isQorLater === false;
        if (hasChanged) {
            deviceEntry.isQorLater = true;
        }
    }
    if (action.platformType !== undefined) {
        if (action.platformType !== deviceEntry.platformType) {
            deviceEntry.platformType = action.platformType;
        }
    }
    if (action.platformLevel !== undefined) {
        if (action.platformLevel !== deviceEntry.platformLevel) {
            deviceEntry.platformLevel = action.platformLevel;
        }
    }
    {
        const effectiveManipulationFlags = action.addedManipulationFlags & device_1.DeviceManipulationFlags.ALL;
        if (effectiveManipulationFlags !== 0) {
            deviceEntry.manipulationFlags = deviceEntry.manipulationFlags | effectiveManipulationFlags;
        }
    }
    await deviceEntry.save({ transaction: cache.transaction });
    if ((0, device_1.hasDeviceManipulation)(deviceEntry)) {
        if (!hadManipulationBefore) {
            await (0, manipulation_1.sendManipulationWarnings)({
                database: cache.database,
                transaction: cache.transaction,
                deviceName: deviceEntry.name,
                familyId: cache.familyId
            });
        }
    }
    cache.invalidiateDeviceList = true;
    cache.incrementTriggeredSyncLevel(1);
}
//# sourceMappingURL=updatedevicestatus.js.map