"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.dispatchMarkTaskPendingAction = dispatchMarkTaskPendingAction;
const taskdone_1 = require("../../../warningmail/taskdone");
const illegal_state_1 = require("../exception/illegal-state");
const missing_item_1 = require("../exception/missing-item");
async function dispatchMarkTaskPendingAction({ action, cache, deviceId }) {
    const taskInfoUnsafe = await cache.database.childTask.findOne({
        where: {
            familyId: cache.familyId,
            taskId: action.taskId
        },
        transaction: cache.transaction,
        attributes: ['categoryId', 'pendingRequest', 'taskTitle']
    });
    if (taskInfoUnsafe === null)
        throw new missing_item_1.MissingTaskException();
    const taskInfo = {
        categoryId: taskInfoUnsafe.categoryId,
        pendingRequest: taskInfoUnsafe.pendingRequest,
        taskTitle: taskInfoUnsafe.taskTitle
    };
    if (taskInfo.pendingRequest !== 0)
        return; // review already requested
    const categoryInfoUnsafe = await cache.database.category.findOne({
        where: {
            familyId: cache.familyId,
            categoryId: taskInfo.categoryId
        },
        attributes: ['childId'],
        transaction: cache.transaction
    });
    if (categoryInfoUnsafe === null) {
        throw new illegal_state_1.IllegalStateException({ staticMessage: 'category referenced from task not found' });
    }
    const categoryInfo = { childId: categoryInfoUnsafe.childId };
    const deviceInfoUnsafe = await cache.database.device.findOne({
        where: {
            familyId: cache.familyId,
            deviceId
        },
        attributes: ['currentUserId'],
        transaction: cache.transaction
    });
    if (deviceInfoUnsafe === null)
        throw new illegal_state_1.SourceDeviceNotFoundException();
    const deviceInfo = { currentUserId: deviceInfoUnsafe.currentUserId };
    if (categoryInfo.childId !== deviceInfo.currentUserId) {
        throw new illegal_state_1.IllegalStateException({ staticMessage: 'Can not mark task pending for other user than the current user' });
    }
    const childInfoUnsafe = await cache.database.user.findOne({
        where: {
            familyId: cache.familyId,
            userId: categoryInfo.childId
        },
        attributes: ['name'],
        transaction: cache.transaction
    });
    if (childInfoUnsafe === null)
        throw new illegal_state_1.SourceUserNotFoundException();
    const childInfo = { name: childInfoUnsafe.name };
    await cache.database.childTask.update({ pendingRequest: 1 }, {
        where: {
            familyId: cache.familyId,
            taskId: action.taskId
        },
        transaction: cache.transaction
    });
    cache.categoriesWithModifiedTasks.add(taskInfo.categoryId);
    cache.incrementTriggeredSyncLevel(1); // parents are not faster reachable
    await (0, taskdone_1.sendTaskDoneMails)({
        database: cache.database,
        transaction: cache.transaction,
        familyId: cache.familyId,
        childName: childInfo.name,
        taskTitle: taskInfo.taskTitle
    });
}
//# sourceMappingURL=marktaskpendingaction.js.map