"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.reportDeviceRemoved = reportDeviceRemoved;
const http_errors_1 = require("http-errors");
const token_1 = require("../../util/token");
const uninstall_1 = require("../warningmail/uninstall");
const websocket_1 = require("../websocket");
async function reportDeviceRemoved({ database, deviceAuthToken, websocket }) {
    await database.transaction(async (transaction) => {
        const deviceEntry = await database.device.findOne({
            where: {
                deviceAuthToken
            },
            transaction
        });
        if (deviceEntry) {
            const currentAuthToken = deviceEntry.deviceAuthToken;
            deviceEntry.didDeviceReportUninstall = true;
            deviceEntry.deviceAuthToken = (0, token_1.generateAuthToken)(); // invalidiate the token
            await deviceEntry.save({ transaction });
            // invalidiate device list
            await database.family.update({
                deviceListVersion: (0, token_1.generateVersionId)()
            }, {
                where: {
                    familyId: deviceEntry.familyId
                },
                transaction
            });
            // add to old devices
            await database.oldDevice.create({
                deviceAuthToken: currentAuthToken
            }, {
                transaction
            });
            await (0, websocket_1.notifyClientsAboutChangesDelayed)({
                database,
                websocket,
                familyId: deviceEntry.familyId,
                sourceDeviceId: null,
                generalLevel: 1,
                targetedLevels: new Map(),
                transaction
            });
            await (0, uninstall_1.sendUninstallWarnings)({
                database,
                familyId: deviceEntry.familyId,
                deviceName: deviceEntry.name,
                transaction
            });
        }
        else {
            const oldDeviceEntry = await database.oldDevice.findOne({
                where: {
                    deviceAuthToken
                },
                transaction
            });
            if (!oldDeviceEntry) {
                throw new http_errors_1.Unauthorized('device not found');
            }
        }
    });
}
//# sourceMappingURL=report-device-removed.js.map