"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createChildRouter = void 0;
const body_parser_1 = require("body-parser");
const express_1 = require("express");
const http_errors_1 = require("http-errors");
const add_device_1 = require("../function/child/add-device");
const logout_at_primary_device_1 = require("../function/child/logout-at-primary-device");
const set_primary_device_1 = require("../function/child/set-primary-device");
const validator_1 = require("./validator");
const createChildRouter = ({ database, websocket, eventHandler }) => {
    const router = (0, express_1.Router)();
    router.post('/add-device', (0, body_parser_1.json)(), async (req, res, next) => {
        try {
            if (!(0, validator_1.isRegisterChildDeviceRequest)(req.body)) {
                throw new http_errors_1.BadRequest();
            }
            const { deviceAuthToken, deviceId, data } = await (0, add_device_1.addChildDevice)({
                request: req.body,
                database,
                eventHandler,
                websocket
            });
            res.json({
                deviceAuthToken,
                ownDeviceId: deviceId,
                data
            });
        }
        catch (ex) {
            next(ex);
        }
    });
    router.post('/update-primary-device', (0, body_parser_1.json)(), async (req, res, next) => {
        try {
            if (!(0, validator_1.isUpdatePrimaryDeviceRequest)(req.body)) {
                throw new http_errors_1.BadRequest();
            }
            const response = await (0, set_primary_device_1.setPrimaryDevice)({
                database,
                deviceAuthToken: req.body.authToken,
                currentUserId: req.body.currentUserId,
                websocket,
                action: req.body.action
            });
            res.json({
                status: response
            });
        }
        catch (ex) {
            next(ex);
        }
    });
    router.post('/logout-at-primary-device', (0, body_parser_1.json)(), async (req, res, next) => {
        try {
            if (!(0, validator_1.isRequestWithAuthToken)(req.body)) {
                throw new http_errors_1.BadRequest();
            }
            await (0, logout_at_primary_device_1.logoutAtPrimaryDevice)({
                deviceAuthToken: req.body.deviceAuthToken,
                database,
                websocket
            });
            res.json({
                ok: true
            });
        }
        catch (ex) {
            next(ex);
        }
    });
    return router;
};
exports.createChildRouter = createChildRouter;
//# sourceMappingURL=child.js.map