"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2021 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAuthRouter = void 0;
const body_parser_1 = require("body-parser");
const express_1 = require("express");
const http_errors_1 = require("http-errors");
const login_by_mail_1 = require("../function/authentication/login-by-mail");
const mail_1 = require("../util/mail");
const validator_1 = require("./validator");
const createAuthRouter = (database) => {
    const router = (0, express_1.Router)();
    router.post('/send-mail-login-code-v2', (0, body_parser_1.json)(), async (req, res, next) => {
        try {
            if (!(0, validator_1.isSendMailLoginCodeRequest)(req.body)) {
                throw new http_errors_1.BadRequest();
            }
            const mail = (0, mail_1.sanitizeMailAddress)(req.body.mail);
            if (!mail) {
                throw new http_errors_1.BadRequest();
            }
            if (!(0, mail_1.isMailAddressCoveredByWhitelist)(mail)) {
                res.json({ mailAddressNotWhitelisted: true });
            }
            else if ((0, mail_1.isMailServerBlacklisted)(mail)) {
                res.json({ mailServerBlacklisted: true });
            }
            else {
                const { mailLoginToken } = await (0, login_by_mail_1.sendLoginCode)({
                    mail,
                    deviceAuthToken: req.body.deviceAuthToken,
                    locale: req.body.locale,
                    database
                });
                res.json({ mailLoginToken });
            }
        }
        catch (ex) {
            next(ex);
        }
    });
    router.post('/sign-in-by-mail-code', (0, body_parser_1.json)(), async (req, res, next) => {
        try {
            if (!(0, validator_1.isSignInByMailCodeRequest)(req.body)) {
                throw new http_errors_1.BadRequest();
            }
            const { mailAuthToken } = await (0, login_by_mail_1.signInByMailCode)({
                receivedCode: req.body.receivedCode,
                mailLoginToken: req.body.mailLoginToken,
                database
            });
            res.json({ mailAuthToken });
        }
        catch (ex) {
            next(ex);
        }
    });
    return router;
};
exports.createAuthRouter = createAuthRouter;
//# sourceMappingURL=auth.js.map