"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2020 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpdateTimelimitRuleAction = void 0;
const minuteofday_1 = require("../util/minuteofday");
const basetypes_1 = require("./basetypes");
const exception_1 = require("./meta/exception");
const util_1 = require("./meta/util");
const actionType = 'UpdateTimelimitRuleAction';
class UpdateTimelimitRuleAction extends basetypes_1.ParentAction {
    constructor({ ruleId, maximumTimeInMillis, dayMask, applyToExtraTimeUsage, start, end, sessionDurationMilliseconds, sessionPauseMilliseconds, perDay }) {
        super();
        this.ruleId = ruleId;
        this.maximumTimeInMillis = maximumTimeInMillis;
        this.dayMask = dayMask;
        this.applyToExtraTimeUsage = applyToExtraTimeUsage;
        this.start = start;
        this.end = end;
        this.sessionDurationMilliseconds = sessionDurationMilliseconds;
        this.sessionPauseMilliseconds = sessionPauseMilliseconds;
        this.perDay = perDay;
        (0, util_1.assertIdWithinFamily)({ actionType, field: 'ruleId', value: ruleId });
        (0, util_1.assertSafeInteger)({ actionType, field: 'maximumTimeInMillis', value: maximumTimeInMillis });
        if (maximumTimeInMillis < 0) {
            (0, util_1.throwOutOfRange)({ actionType, field: 'maximumTimeInMillis', value: maximumTimeInMillis });
        }
        (0, util_1.assertSafeInteger)({ actionType, field: 'dayMask', value: dayMask });
        if (dayMask < 0 || dayMask > (1 | 2 | 4 | 8 | 16 | 32 | 64)) {
            (0, util_1.throwOutOfRange)({ actionType, field: 'dayMask', value: dayMask });
        }
        (0, util_1.assertSafeInteger)({ actionType, field: 'start', value: start });
        (0, util_1.assertSafeInteger)({ actionType, field: 'end', value: end });
        (0, util_1.assertSafeInteger)({ actionType, field: 'sessionDurationMilliseconds', value: sessionDurationMilliseconds });
        (0, util_1.assertSafeInteger)({ actionType, field: 'sessionPauseMilliseconds', value: sessionPauseMilliseconds });
        if (start < minuteofday_1.MinuteOfDay.MIN || end > minuteofday_1.MinuteOfDay.MAX || start > end) {
            throw new exception_1.InvalidActionParameterException({
                actionType,
                staticMessage: 'time slot out of range'
            });
        }
        if (sessionDurationMilliseconds < 0 || sessionPauseMilliseconds < 0) {
            throw new exception_1.InvalidActionParameterException({
                actionType,
                staticMessage: 'session duration lesser than zero'
            });
        }
    }
}
exports.UpdateTimelimitRuleAction = UpdateTimelimitRuleAction;
UpdateTimelimitRuleAction.parse = ({ ruleId, time, days, extraTime, start, end, dur, pause, perDay }) => (new UpdateTimelimitRuleAction({
    ruleId,
    maximumTimeInMillis: time,
    dayMask: days,
    applyToExtraTimeUsage: extraTime,
    start: start !== null && start !== void 0 ? start : minuteofday_1.MinuteOfDay.MIN,
    end: end !== null && end !== void 0 ? end : minuteofday_1.MinuteOfDay.MAX,
    sessionDurationMilliseconds: dur !== null && dur !== void 0 ? dur : 0,
    sessionPauseMilliseconds: pause !== null && pause !== void 0 ? pause : 0,
    perDay: perDay !== null && perDay !== void 0 ? perDay : false
}));
//# sourceMappingURL=updatetimelimitrule.js.map