"use strict";
/*
 * server component for the TimeLimit App
 * Copyright (C) 2019 - 2020 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.InvalidChangeParentPasswordIntegrityException = exports.ChangeParentPasswordAction = void 0;
const crypto_1 = require("crypto");
const basetypes_1 = require("./basetypes");
const exception_1 = require("./meta/exception");
const util_1 = require("./meta/util");
const actionType = 'ChangeParentPasswordAction';
class ChangeParentPasswordAction extends basetypes_1.ParentAction {
    constructor({ parentUserId, newPasswordFirstHash, newPasswordSecondSalt, newPasswordSecondHashEncrypted, integrity }) {
        super();
        (0, util_1.assertIdWithinFamily)({ actionType, field: 'parentUserId', value: parentUserId });
        if ((!parentUserId) ||
            (!newPasswordFirstHash) ||
            (!newPasswordSecondSalt) ||
            (!newPasswordSecondHashEncrypted) ||
            (!integrity)) {
            throw new exception_1.InvalidActionParameterException({
                actionType,
                staticMessage: 'missing required parameter for change parent password'
            });
        }
        if (integrity.length !== 128) {
            throw new exception_1.InvalidActionParameterException({
                actionType,
                staticMessage: 'wrong length of integrity data'
            });
        }
        (0, util_1.assertHexString)({ actionType, field: 'newPasswordSecondHashEncrypted', value: newPasswordSecondHashEncrypted });
        (0, util_1.assertHexString)({ actionType, field: 'integrity', value: integrity });
        if (newPasswordSecondHashEncrypted.length <= 70) {
            throw new exception_1.InvalidActionParameterException({ actionType, staticMessage: 'wrong length of the new password' });
        }
        this.parentUserId = parentUserId;
        this.newPasswordFirstHash = newPasswordFirstHash;
        this.newPasswordSecondSalt = newPasswordSecondSalt;
        this.newPasswordSecondHashEncrypted = newPasswordSecondHashEncrypted;
        this.integrity = integrity;
    }
    assertIntegrityValid({ oldPasswordSecondHash }) {
        const integrityData = oldPasswordSecondHash +
            this.parentUserId +
            this.newPasswordFirstHash +
            this.newPasswordSecondSalt +
            this.newPasswordSecondHashEncrypted;
        const expected = (0, crypto_1.createHash)('sha512').update(integrityData).digest('hex');
        if (expected !== this.integrity) {
            throw new InvalidChangeParentPasswordIntegrityException();
        }
    }
    decryptSecondHash({ oldPasswordSecondHash }) {
        const ivHex = this.newPasswordSecondHashEncrypted.substring(0, 32);
        const salt = this.newPasswordSecondHashEncrypted.substring(32, 64);
        const encryptedData = this.newPasswordSecondHashEncrypted.substring(64);
        const keyData = oldPasswordSecondHash + salt;
        const key = (0, crypto_1.createHash)('sha512').update(keyData).digest().slice(0, 16);
        const decipher = (0, crypto_1.createDecipheriv)('aes-128-ctr', key, Buffer.from(ivHex, 'hex'));
        decipher.setAutoPadding(false);
        const decryptedSecondHash = decipher.update(Buffer.from(encryptedData, 'hex')).toString() + decipher.final().toString();
        return decryptedSecondHash;
    }
}
exports.ChangeParentPasswordAction = ChangeParentPasswordAction;
ChangeParentPasswordAction.parse = ({ userId, hash, secondSalt, secondHashEncrypted, integrity }) => (new ChangeParentPasswordAction({
    parentUserId: userId,
    newPasswordFirstHash: hash,
    newPasswordSecondSalt: secondSalt,
    newPasswordSecondHashEncrypted: secondHashEncrypted,
    integrity
}));
class InvalidChangeParentPasswordIntegrityException extends Error {
    constructor() { super('invalid integrity for change parent password action'); }
}
exports.InvalidChangeParentPasswordIntegrityException = InvalidChangeParentPasswordIntegrityException;
//# sourceMappingURL=changeparentpassword.js.map