#!/usr/bin/env python3
import json
import os
import shutil
import tempfile
from collections.abc import AsyncIterator
from contextlib import asynccontextmanager
from pathlib import Path
from typing import Any

import requests
from mcp.server.fastmcp import FastMCP
from mcp.server.fastmcp.prompts.base import Prompt
from mcp.server.fastmcp.resources.types import FunctionResource
from mcp.server.fastmcp.server import Context
from mcp.shared.exceptions import McpError
from mcp.types import ErrorData
from mcp.types import INTERNAL_ERROR
from mcp.types import INVALID_PARAMS
from opentelemetry.trace.propagation import (
    get_current_span,
)
from pydantic import Field
from pydantic import ValidationError
from starlette.requests import Request
from starlette.responses import JSONResponse
from starlette.routing import Route

from semgrep import __VERSION__
from semgrep.mcp.models import CodeFile
from semgrep.mcp.models import CodePath
from semgrep.mcp.models import Finding
from semgrep.mcp.models import FindingElicitationSchema
from semgrep.mcp.models import SemgrepScanResult
from semgrep.mcp.semgrep import mk_context
from semgrep.mcp.semgrep import run_semgrep_output
from semgrep.mcp.semgrep import run_semgrep_process_sync
from semgrep.mcp.semgrep import run_semgrep_via_rpc
from semgrep.mcp.semgrep import SemgrepContext
from semgrep.mcp.utilities.tracing import attach_findings_metrics
from semgrep.mcp.utilities.tracing import attach_scan_metrics
from semgrep.mcp.utilities.tracing import start_tracing
from semgrep.mcp.utilities.tracing import with_span
from semgrep.mcp.utilities.tracing import with_tool_span
from semgrep.mcp.utilities.utils import findings_elicitation_enabled
from semgrep.mcp.utilities.utils import get_identity
from semgrep.mcp.utilities.utils import get_semgrep_api_url
from semgrep.mcp.utilities.utils import get_semgrep_app_token
from semgrep.mcp.utilities.utils import is_hosted
from semgrep.mcp.utilities.utils import re_identity_string
from semgrep.metrics import Finding as MetricsFinding
from semgrep.semgrep_interfaces.semgrep_output_v1 import CliOutput
from semgrep.state import get_state
from semgrep.verbose_logging import getLogger

logger = getLogger(__name__)

# ---------------------------------------------------------------------------------
# Constants
# ---------------------------------------------------------------------------------

SEMGREP_API_VERSION = "v1"

# Field definitions for function parameters
REMOTE_CODE_FILES_FIELD = Field(
    description="List of dictionaries with 'path' and 'content' keys"
)
LOCAL_CODE_FILES_FIELD = Field(
    description=(
        "List of dictionaries with 'path' pointing to the absolute path of the code file"
    )
)

CONFIG_FIELD = Field(
    description="Optional Semgrep configuration string (e.g. 'p/docker', 'p/xss', 'auto')",
    default=None,
)

RULE_FIELD = Field(description="Semgrep YAML rule string")
RULE_ID_FIELD = Field(description="Semgrep rule ID")

CODE_FIELD = Field(description="The code to get the AST for")
LANGUAGE_FIELD = Field(description="The programming language of the code")

# ---------------------------------------------------------------------------------
# Utilities
# ---------------------------------------------------------------------------------


def safe_join(base_dir: str, untrusted_path: str) -> str:
    """
    Joins a base directory with an untrusted relative path and ensures the final path
    doesn't escape the base directory.

    Args:
        base_dir: The base directory to join the untrusted path to
        untrusted_path: The untrusted relative path to join to the base directory
    """
    # Absolute, normalized path to the base directory
    base_path = Path(base_dir).resolve()

    # Handle empty path, current directory, or paths with only slashes
    if not untrusted_path or untrusted_path == "." or untrusted_path.strip("/") == "":
        return base_path.as_posix()

    # Ensure untrusted path is not absolute
    # This is soft validation, path traversal is checked later
    if Path(untrusted_path).is_absolute():
        raise ValueError("Untrusted path must be relative")

    # Join and normalize the untrusted path
    full_path = base_path / Path(untrusted_path)

    # Ensure the final path doesn't escape the base directory
    if not full_path == full_path.resolve():
        raise ValueError(
            f"Untrusted path escapes the base directory!: {untrusted_path}"
        )

    return full_path.as_posix()


# Path validation
def validate_absolute_path(path_to_validate: str, param_name: str) -> str:
    """Validates an absolute path to ensure it's safe to use"""
    if not Path(path_to_validate).is_absolute():
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS,
                message=f"{param_name} must be an absolute path. Received: {path_to_validate}",
            )
        )

    # Normalize path and ensure no path traversal is possible
    normalized_path = os.path.normpath(path_to_validate)

    # Check if normalized path is still absolute
    if not Path(normalized_path).resolve() == Path(normalized_path):
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS,
                message=f"{param_name} contains invalid path traversal sequences",
            )
        )

    return normalized_path


def validate_config(config: str | None = None) -> str:
    """Validates semgrep configuration parameter"""
    # Allow registry references (p/ci, p/security, etc.)
    if (
        config is None
        or config.startswith("p/")
        or config.startswith("r/")
        or config == "auto"
    ):
        return config or ""
    # Otherwise, treat as path and validate
    return validate_absolute_path(config, "config")


# Utility functions for handling code content
def create_temp_files_from_code_content(code_files: list[CodeFile]) -> str:
    """
    Creates temporary files from code content

    Args:
        code_files: List of CodeFile objects

    Returns:
        Path to temporary directory containing the files

    Raises:
        McpError: If there are issues creating or writing to files
    """
    temp_dir = None

    try:
        # Create a temporary directory
        temp_dir = tempfile.mkdtemp(prefix="semgrep_scan_")

        # Create files in the temporary directory
        for file_info in code_files:
            filename = file_info.path
            if not filename:
                continue

            temp_file_path = safe_join(temp_dir, filename)

            try:
                # Create subdirectories if needed
                os.makedirs(os.path.dirname(temp_file_path), exist_ok=True)

                # Write content to file
                with open(temp_file_path, "w") as f:
                    f.write(file_info.content)
            except OSError as e:
                raise McpError(
                    ErrorData(
                        code=INTERNAL_ERROR,
                        message=f"Failed to create or write to file {filename}: {e!s}",
                    )
                ) from e

        return temp_dir
    except Exception as e:
        if temp_dir:
            # Clean up temp directory if creation failed
            shutil.rmtree(temp_dir, ignore_errors=True)
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Failed to create temporary files: {e!s}"
            )
        ) from e


def get_semgrep_scan_args(temp_dir: str, config: str | None = None) -> list[str]:
    """
    Builds command arguments for semgrep scan

    Args:
        temp_dir: Path to temporary directory containing the files
        config: Optional Semgrep configuration (e.g. "auto" or absolute path to rule file)

    Returns:
        List of command arguments
    """

    # Build command arguments and just run semgrep scan
    # if no config is provided to allow for either the default "auto"
    # or whatever the logged in config is
    args = ["scan", "--json", "--experimental"]  # avoid the extra exec
    args.extend(["--x-mcp"])
    if config:
        args.extend(["--config", config])
    args.append(temp_dir)
    return args


def validate_local_files(local_files: list[CodePath]) -> list[CodeFile]:
    """
    Validates the local_files parameter for semgrep scan using Pydantic validation

    Args:
        local_files: List of singleton dictionaries with a "path" key

    Raises:
        McpError: If validation fails
    """
    if not local_files:
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS,
                message="local_files must be a non-empty list of file objects",
            )
        )
    try:
        # Pydantic will automatically validate each item in the list
        validated_local_files = []
        for file in local_files:
            path = file.path
            if not Path(path).is_absolute():
                raise McpError(
                    ErrorData(
                        code=INVALID_PARAMS,
                        message="code_files.path must be a absolute path",
                    )
                )
            contents = Path(path).read_text()
            # We need to not use the absolute path here, as there is logic later
            # that raises, to prevent path traversal.
            # In reality, the name of the file is pretty immaterial. We only
            # want the accurate path insofar as we can get the contents (whcih we do here)
            # and so we can remember what original file it corresponds to.
            # Taking the name of the file should be enough.
            validated_local_files.append(
                CodeFile(path=Path(path).name, content=contents)
            )
    except Exception as e:
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS, message=f"Invalid local code files format: {e!s}"
            )
        ) from e

    return validated_local_files


def validate_remote_files(code_files: list[CodeFile]) -> list[CodeFile]:
    """
    Validates the code_files parameter for semgrep scan using Pydantic validation

    Args:
        code_files: List of dictionaries with a "path" and "content" key

    Raises:
        McpError: If validation fails
    """
    if not code_files:
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS,
                message="code_files must be a non-empty list of file objects",
            )
        )
    try:
        # Pydantic will automatically validate each item in the list
        validated_code_files = [CodeFile.model_validate(file) for file in code_files]

        return validated_code_files
    except Exception as e:
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS, message=f"Invalid remote code files format: {e!s}"
            )
        ) from e


def remove_temp_dir_from_results(results: SemgrepScanResult, temp_dir: str) -> None:
    """
    Clean the results from semgrep by converting temporary file paths back to
    original relative paths

    Args:
        results: SemgrepScanResult object containing semgrep results
        temp_dir: Path to temporary directory used for scanning
    """
    # Process findings results
    for finding in results.results:
        if "path" in finding:
            try:
                finding["path"] = os.path.relpath(finding["path"], temp_dir)
            except ValueError:
                # Skip if path is not relative to temp_dir
                continue

    # Process scanned paths
    if "scanned" in results.paths:
        results.paths["scanned"] = [
            os.path.relpath(path, temp_dir) for path in results.paths["scanned"]
        ]

    if "skipped" in results.paths:
        results.paths["skipped"] = [
            os.path.relpath(path, temp_dir) for path in results.paths["skipped"]
        ]


async def get_workspace_dir(ctx: Context) -> str | None:
    """
    Get the workspace directory from the context

    Note: We must invoke this method at request time, and not lifespan time,
    because it relies on the `ctx.request_context`, which does not exist
    when we initialize the server.
    """
    # This step fails when we are running tests, so I am wrapping it in a try/except
    try:
        # This URI is supposed to begin with `file://`
        roots = await ctx.request_context.session.list_roots()
        logger.debug(f"Got roots from client: {roots}")

        # Just to be safe. It's probably impossible.
        if len(roots.roots) == 0:
            logger.warning("Somehow, no roots found")
            return None

        uri: str = str(roots.roots[0].uri)
        path = uri[7:] if uri.startswith("file://") else uri

        logger.debug(f"Determined path of workspace directory: {path}")

        return path
    except Exception:
        return ""


async def finding_elicitation(
    ctx: Context, results: SemgrepScanResult
) -> tuple[
    list[tuple[str, MetricsFinding]],
    list[tuple[str, MetricsFinding]],
    list[tuple[str, MetricsFinding]],
]:
    """
    Elicit user input for findings to determine if they are true positives or false positives.
    If findings elicitation is not enabled, throws an error.

    Args:
        ctx: Context object
        results: SemgrepScanResult object

    Returns:
        Tuple of lists of true positive (first item), false positive (second item), and skipped (third item) findings
    """
    with with_span(get_current_span(), "finding_elicitation") as _:
        findings = get_state().metrics.cli_matches_to_findings(results.results)
        if not findings_elicitation_enabled():
            raise McpError(
                ErrorData(
                    code=INVALID_PARAMS,
                    message="Findings elicitation is not enabled.",
                )
            )

        true_positive_findings, false_positive_findings, skipped_findings = [], [], []
        logger.info(f"Findings to elicit: {findings}")
        for rule_id, finding in findings:
            result = await ctx.elicit(
                message=f"🔍 Semgrep Finding Detected: [{finding.severity}] {rule_id} ({finding.path}:{finding.line}:{finding.col})",
                schema=FindingElicitationSchema,
            )
            if result.action == "accept" and result.data:
                if result.data.true_positive:
                    logger.info(f"Finding {finding} is a true positive.")
                    true_positive_findings.append((rule_id, finding))
                elif result.data.false_positive:
                    logger.info(f"Finding {finding} is a false positive.")
                    false_positive_findings.append((rule_id, finding))
                elif result.data.skip:
                    logger.info(f"Finding {finding} is a skip.")
                    skipped_findings.append((rule_id, finding))

        return true_positive_findings, false_positive_findings, skipped_findings


# ---------------------------------------------------------------------------------
# MCP Server
# ---------------------------------------------------------------------------------


@asynccontextmanager
async def server_lifespan(_server: FastMCP) -> AsyncIterator[SemgrepContext]:
    """Manage server startup and shutdown lifecycle."""
    # Initialize resources on startup with tracing
    # MCP requires Pro Engine
    with start_tracing("mcp-python-server") as span:
        context = await mk_context(top_level_span=span)

        try:
            yield context
        finally:
            context.shutdown()


# ---------------------------------------------------------------------------------
# MCP Tools
# ---------------------------------------------------------------------------------


@with_tool_span(is_semgrep_scan=False)
async def semgrep_rule_schema(ctx: Context) -> str:
    """
    Get the schema for a Semgrep rule

    Use this tool when you need to:
      - get the schema required to write a Semgrep rule
      - need to see what fields are available for a Semgrep rule
      - verify what fields are available for a Semgrep rule
      - verify the syntax for a Semgrep rule is correct
    """
    try:
        response = requests.get(
            f"{(get_semgrep_api_url())}/schema_url", timeout=(2, 30)
        )
        response.raise_for_status()
        data: dict[str, str] = response.json()
        schema_url: str = data["schema_url"]
        response = requests.get(schema_url, timeout=(2, 30))
        response.raise_for_status()
        return str(response.text)
    except Exception as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR,
                message=f"Error getting schema for Semgrep rule: {e!s}",
            )
        ) from e


@with_tool_span(is_semgrep_scan=False)
async def get_supported_languages(ctx: Context) -> list[str]:
    """
    Returns a list of supported languages by Semgrep

    Only use this tool if you are not sure what languages Semgrep supports.
    """
    args = ["show", "supported-languages", "--experimental"]

    # Parse output and return list of languages
    languages = await run_semgrep_output(top_level_span=None, args=args)
    return [lang.strip() for lang in languages.strip().split("\n") if lang.strip()]


async def get_deployment_slug() -> str:
    """
    Fetches and caches the deployment slug from Semgrep API.

    Returns:
        str: The deployment slug

    Raises:
        McpError: If unable to fetch deployments or no deployments found
    """

    # Get API token
    api_token = get_semgrep_app_token()
    if not api_token:
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS,
                message="""
                  SEMGREP_APP_TOKEN environment variable must be set or user
                  must be logged in to use this tool
                """,
            )
        )

    # Fetch deployments
    url = f"{get_semgrep_api_url()}/v1/deployments"
    headers = {"Authorization": f"Bearer {api_token}", "Accept": "application/json"}

    try:
        response = requests.get(url, headers=headers, timeout=(2, 30))
        response.raise_for_status()
        data = response.json()

        # Extract deployment slug - assuming we want the first deployment
        deployments = data.get("deployments", [])
        if not deployments or not deployments[0].get("slug"):
            raise McpError(
                ErrorData(
                    code=INTERNAL_ERROR,
                    message="No deployments found for this API token",
                )
            )

        return str(deployments[0]["slug"])

    except requests.exceptions.HTTPError as e:
        if e.response.status_code == 401:
            raise McpError(
                ErrorData(
                    code=INVALID_PARAMS,
                    message="Invalid API token: check your SEMGREP_APP_TOKEN environment variable.",
                )
            ) from e
        else:
            raise McpError(
                ErrorData(
                    code=INTERNAL_ERROR,
                    message=f"Error fetching deployments: {e.response.text}",
                )
            ) from e
    except Exception as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR,
                message=f"Error fetching deployments from Semgrep: {e!s}",
            )
        ) from e


@with_tool_span(is_semgrep_scan=False)
async def semgrep_findings(
    ctx: Context,
    issue_type: str = "sast",  # noqa: B006
    repos: list[str] | None = None,  # pyright: ignore  # noqa: RUF013
    status: str = "open",
    severities: list[str] | None = None,  # pyright: ignore  # noqa: RUF013
    confidence: list[str] | None = None,  # pyright: ignore  # noqa: RUF013
    autotriage_verdict: str = "true_positive",
    page: int = 0,
    page_size: int = 100,
) -> list[Finding]:
    """
    Fetches findings from the Semgrep AppSec Platform Findings API.

    This function retrieves security, code quality, and supply chain findings that have already been
    identified by previous Semgrep scans and uploaded to the Semgrep AppSec platform. It does NOT
    perform a new scan or analyze code directly. Instead, it queries the Semgrep API to access
    historical scan results for a given repository or set of repositories.

    DEFAULT BEHAVIOR: By default, this tool should filter by the current repository. The model
    should determine the current repository name and pass it in the 'repos' parameter to ensure
    findings are scoped to the relevant codebase. However, users may explicitly request findings
    from other repositories, in which case the model should respect that request.

    Use this function when a prompt requests a summary, list, or analysis of existing findings,
    such as:
        - "Please list the top 10 security findings and propose solutions for them."
        - "Show all open critical vulnerabilities in this repository."
        - "Summarize the most recent Semgrep scan results."
        - "Get findings from repository X" (explicitly requesting different repo)

    This function is ideal for:
    - Reviewing, listing, or summarizing findings from past scans.
    - Providing actionable insights or remediation advice based on existing scan data.

    Do NOT use this function to perform a new scan or check code that has not yet been analyzed by
    Semgrep. For new scans, use the appropriate scanning function.

    Args:
        issue_type (str): Filter findings by type. Use 'sast' for code analysis
            findings and 'sca' for supply chain analysis findings (e.g., 'sast', 'sca').
        status (Optional[str]): Filter findings by status (default: 'open').
        repos (Optional[List[str]]): List of repository names to filter results. By default, should
            include the current repository name to scope findings appropriately. Can be overridden
            when users explicitly request findings from other repositories.
        severities (Optional[List[str]]): Filter findings by severity (e.g., ['critical', 'high']).
        confidence (Optional[List[str]]): Filter findings by confidence level (e.g., ['high']).
        autotriage_verdict (Optional[str]): Filter findings by auto-triage verdict
            (default: 'true_positive').
        page (Optional[int]): Page number for paginated results. (default: 0)
        page_size (int): Number of findings per page (default: 100, min: 100, max: 3000).

    Returns:
        List[Finding]: A list of findings matching the specified filters, where each finding
        contains details such as rule ID, description, severity, file location, and remediation
        guidance if available.
    """
    allowed_issue_types = {"sast", "sca"}
    if issue_type not in allowed_issue_types:
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS,
                message=f"Invalid issue_type: {issue_type}. Allowed values are 'sast' or 'sca'.",
            )
        )

    if not (100 <= page_size <= 3000):
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS, message="page_size must be between 100 and 3000."
            )
        )

    # Check whether the token has the `webapi` role
    identity = await get_identity()
    match = re_identity_string.search(identity["identity"])
    if match is None:
        logger.error("Identity string in unexpected format")
    else:
        inner = match.group(1)
        if "webapi" not in inner:
            raise McpError(
                ErrorData(
                    code=INVALID_PARAMS,
                    message="Cannot access findings without token with `webapi` role: user must generate one manually from `semgrep.dev`",
                )
            )

    # If the token is good, let's get the deployment info.
    deployment = await get_deployment_slug()

    api_token = get_semgrep_app_token()
    if not api_token:
        raise McpError(
            ErrorData(
                code=INVALID_PARAMS,
                message="SEMGREP_APP_TOKEN environment variable must be set to use this tool. "
                "Create a token at semgrep.dev to continue.",
            )
        )

    url = f"https://semgrep.dev/api/v1/deployments/{deployment}/findings"
    headers = {"Authorization": f"Bearer {api_token}", "Accept": "application/json"}

    params_to_filter: dict[str, Any] = {
        "issue_type": issue_type,
        "status": status,
        "repos": ",".join(repos) if repos else None,
        "severities": severities,
        "confidence": confidence,
        "autotriage_verdict": autotriage_verdict,
        "page": page,
        "page_size": page_size,
    }
    params = {k: v for k, v in params_to_filter.items() if v is not None}

    try:
        response = requests.get(url, headers=headers, params=params, timeout=(2, 30))
        response.raise_for_status()
        data = response.json()
        return [Finding.model_validate(finding) for finding in data.get("findings", [])]
    except requests.exceptions.HTTPError as e:
        if e.response.status_code == 401:
            raise McpError(
                ErrorData(
                    code=INVALID_PARAMS,
                    message="Invalid API token: check your SEMGREP_APP_TOKEN environment variable.",
                )
            ) from e
        elif e.response.status_code == 404:
            raise McpError(
                ErrorData(
                    code=INVALID_PARAMS,
                    message=f"Deployment '{deployment}' not found or you don't have access to it.",
                )
            ) from e
        else:
            raise McpError(
                ErrorData(
                    code=INTERNAL_ERROR,
                    message=f"Error fetching findings: {e.response.text}",
                )
            ) from e
    except ValidationError as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Error parsing semgrep output: {e!s}"
            )
        ) from e
    except Exception as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR,
                message=f"Error fetching findings from Semgrep: {e!s}",
            )
        ) from e


@with_tool_span()
async def semgrep_scan_with_custom_rule(
    ctx: Context,
    code_files: list[CodeFile] = REMOTE_CODE_FILES_FIELD,
    rule: str = RULE_FIELD,
) -> SemgrepScanResult:
    """
    Runs a Semgrep scan with a custom rule on provided code content
    and returns the findings in JSON format

    Use this tool when you need to:
      - scan code files for specific security vulnerability not covered by the default Semgrep rules
      - scan code files for specific issue not covered by the default Semgrep rules
    """

    workspace_dir = await get_workspace_dir(ctx)

    # Validate code_files
    validated_code_files = validate_remote_files(code_files)
    temp_dir = None
    try:
        # Create temporary files from code content
        temp_dir = create_temp_files_from_code_content(validated_code_files)
        # Write rule to file
        rule_file_path = os.path.join(temp_dir, "rule.yaml")
        with open(rule_file_path, "w") as f:
            f.write(rule)

        # Run semgrep scan with custom rule
        args = get_semgrep_scan_args(temp_dir, rule_file_path)
        output = await run_semgrep_output(top_level_span=None, args=args)
        results: SemgrepScanResult = SemgrepScanResult.model_validate_json(output)

        if findings_elicitation_enabled():
            tps, fps, skips = await finding_elicitation(ctx, results)
            attach_findings_metrics(get_current_span(), tps, fps, skips)
        else:
            logger.info("Findings elicitation is not enabled, skipping.")

        attach_scan_metrics(
            get_current_span(), results, workspace_dir, validated_code_files
        )

        remove_temp_dir_from_results(results, temp_dir)
        return results

    except McpError as e:
        raise e
    except ValidationError as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Error parsing semgrep output: {e!s}"
            )
        ) from e
    except Exception as e:
        raise McpError(
            ErrorData(code=INTERNAL_ERROR, message=f"Error running semgrep scan: {e!s}")
        ) from e

    finally:
        if temp_dir:
            # Clean up temporary files
            shutil.rmtree(temp_dir, ignore_errors=True)


@with_tool_span(is_semgrep_scan=False)
async def get_abstract_syntax_tree(
    ctx: Context,
    code: str = CODE_FIELD,
    language: str = LANGUAGE_FIELD,
) -> str:
    """
    Returns the Abstract Syntax Tree (AST) for the provided code file in JSON format

    Use this tool when you need to:
      - get the Abstract Syntax Tree (AST) for the provided code file\
      - get the AST of a file
      - understand the structure of the code in a more granular way
      - see what a parser sees in the code
    """
    temp_dir = None
    temp_file_path = ""
    try:
        # Create temporary directory and file for AST generation
        temp_dir = tempfile.mkdtemp(prefix="semgrep_ast_")
        temp_file_path = os.path.join(temp_dir, "code.txt")  # safe

        # Write content to file
        with open(temp_file_path, "w") as f:
            f.write(code)

        args = [
            "--experimental",
            "--dump-ast",
            "-l",
            language,
            "--json",
            temp_file_path,
        ]
        return await run_semgrep_output(top_level_span=None, args=args)

    except McpError as e:
        raise e
    except ValidationError as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Error parsing semgrep output: {e!s}"
            )
        ) from e
    except OSError as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR,
                message=f"Failed to create or write to file {temp_file_path}: {e!s}",
            )
        ) from e
    except Exception as e:
        raise McpError(
            ErrorData(code=INTERNAL_ERROR, message=f"Error running semgrep scan: {e!s}")
        ) from e
    finally:
        if temp_dir:
            # Clean up temporary files
            shutil.rmtree(temp_dir, ignore_errors=True)


# ---------------------------------------------------------------------------------
# Supply Chain scanning
# ---------------------------------------------------------------------------------


async def semgrep_scan_sca(
    context: SemgrepContext,
    workspace_dir: str,
) -> CliOutput:
    cwd = os.getcwd()

    # Do this from the repo so we only scan stuff in there
    os.chdir(workspace_dir)
    args = ["scan", "--config", "supply-chain", "--json", "--x-mcp"]
    output = await run_semgrep_process_sync(context.top_level_span, args)
    os.chdir(cwd)

    resp_json = json.loads(output.stdout.decode())

    return CliOutput.from_json(resp_json)


@with_tool_span()
async def semgrep_scan_supply_chain(
    ctx: Context,
) -> CliOutput:
    """
    Runs a Semgrep supply chain scan on the provided workspace directory,
    to identify potential third-party security vulnerabilities.

    Use this tool when you:
      - change the version of a dependency in a project
      - add a new dependency to a project
      - update the lockfiles of a project
    """

    context: SemgrepContext = ctx.request_context.lifespan_context
    workspace_dir = await get_workspace_dir(ctx)
    if workspace_dir is None:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR,
                message="Workspace directory not found",
            )
        )

    try:
        if context.process is None:
            raise McpError(
                ErrorData(
                    code=INTERNAL_ERROR,
                    message="Supply Chain scan requires an active Semgrep daemon to be running.",
                )
            )
        else:
            logger.info(f"Running Supply Chain scan on path: {workspace_dir}")
            return await semgrep_scan_sca(context, workspace_dir)
    except McpError as e:
        raise e
    except ValidationError as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Error parsing semgrep output: {e!s}"
            )
        ) from e
    except Exception as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR,
                message=f"Error running semgrep scan (supply chain): {e!s}",
            )
        ) from e


# ---------------------------------------------------------------------------------
# Scanning tools
# ---------------------------------------------------------------------------------


@with_tool_span()
async def semgrep_scan_cli(
    ctx: Context,
    workspace_dir: str | None,
    code_files: list[CodeFile],
) -> SemgrepScanResult:
    """
    Runs a Semgrep scan on provided code content and returns the findings in JSON format

    Depending on whether `USE_SEMGREP_RPC` is set, this tool will either run a `pysemgrep`
    CLI scan, or an RPC-based scan.

    Respectively, this will cause us to return either a `SemgrepScanResult` or a `CliOutput`.

    Use this tool when you need to:
      - scan code files for security vulnerabilities
      - scan code files for other issues
    """

    temp_dir = None
    try:
        # Create temporary files from code content
        temp_dir = create_temp_files_from_code_content(code_files)
        args = get_semgrep_scan_args(temp_dir, None)
        output = await run_semgrep_output(top_level_span=None, args=args)
        results: SemgrepScanResult = SemgrepScanResult.model_validate_json(output)
        remove_temp_dir_from_results(results, temp_dir)

        if findings_elicitation_enabled():
            tps, fps, skips = await finding_elicitation(ctx, results)
            attach_findings_metrics(get_current_span(), tps, fps, skips)
        else:
            logger.info("Findings elicitation is not enabled, skipping.")

        attach_scan_metrics(get_current_span(), results, workspace_dir, code_files)

        return results

    except McpError as e:
        raise e
    except ValidationError as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Error parsing semgrep output: {e!s}"
            )
        ) from e
    except Exception as e:
        raise McpError(
            ErrorData(code=INTERNAL_ERROR, message=f"Error running semgrep scan: {e!s}")
        ) from e

    finally:
        if temp_dir:
            # Clean up temporary files
            shutil.rmtree(temp_dir, ignore_errors=True)


@with_tool_span(
    send_metrics=False
)  # Don't send metrics for this because we send metrics for each RPC call from the daemon
async def semgrep_scan_rpc(
    ctx: Context,
    workspace_dir: str | None,
    code_files: list[CodeFile],
) -> SemgrepScanResult:
    """
    Runs a Semgrep scan on provided code content using the new Semgrep RPC feature.

    This should run much faster than the comparative `semgrep_scan` tool.
    """

    temp_dir = None
    try:
        # TODO: perhaps should return more interpretable results?
        context: SemgrepContext = ctx.request_context.lifespan_context
        results = await run_semgrep_via_rpc(context, workspace_dir, code_files)

        if findings_elicitation_enabled():
            tps, fps, skips = await finding_elicitation(ctx, results)
            attach_findings_metrics(get_current_span(), tps, fps, skips)
        else:
            logger.info("Findings elicitation is not enabled, skipping.")

        attach_scan_metrics(get_current_span(), results, workspace_dir, code_files)

        return results
    except McpError as e:
        raise e
    except ValidationError as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Error parsing semgrep output: {e!s}"
            )
        ) from e
    except Exception as e:
        raise McpError(
            ErrorData(code=INTERNAL_ERROR, message=f"Error running semgrep scan: {e!s}")
        ) from e

    finally:
        if temp_dir:
            # Clean up temporary files
            shutil.rmtree(temp_dir, ignore_errors=True)


async def semgrep_scan_core(
    ctx: Context,
    workspace_dir: str | None,
    code_files: list[CodeFile],
) -> SemgrepScanResult:
    """
    Runs a Semgrep scan on provided CodeFile objects and returns the findings in JSON format

    Depending on whether `USE_SEMGREP_RPC` is set, this tool will either run a `pysemgrep`
    CLI scan, or an RPC-based scan.

    Respectively, this will cause us to return either a `SemgrepScanResult` or a `CliOutput`.
    """

    context: SemgrepContext = ctx.request_context.lifespan_context

    paths = [cf.path for cf in code_files]

    if context.process is not None:
        logger.info(f"Running RPC-based scan on paths: {paths}")
        return await semgrep_scan_rpc(ctx, workspace_dir, code_files)
    else:
        logger.info(f"Running CLI-based scan on paths: {paths}")
        return await semgrep_scan_cli(ctx, workspace_dir, code_files)


@with_tool_span()
async def semgrep_scan_remote(
    ctx: Context,
    code_files: list[CodeFile] = REMOTE_CODE_FILES_FIELD,
) -> SemgrepScanResult:
    """
    Runs a Semgrep scan on provided code content and returns the findings in JSON format

    Use this tool when you need to:
      - scan code files for security vulnerabilities
      - scan code files for other issues
    """

    # Implementer's note:
    # This is one possible entry point for regular scanning, depending on whether
    # the server is remotely hosted or not.
    # If the server is hosted, only this tool will be available, and not the
    # `semgrep_scan` tool.

    validated_code_files = validate_remote_files(code_files)

    return await semgrep_scan_core(ctx, None, validated_code_files)


@with_tool_span(
    send_metrics=False
)  # Don't send metrics for this because it is a wrapper for either the rpc scan or the cli scan
async def semgrep_scan(
    ctx: Context,
    code_files: list[CodePath] = LOCAL_CODE_FILES_FIELD,
) -> SemgrepScanResult:
    """
    Runs a Semgrep scan locally on provided code files returns the findings in JSON format.

    Files are expected to be absolute paths to the code files.

    Use this tool when you need to:
      - scan code files for security vulnerabilities
      - scan code files for other issues
    """

    workspace_dir = await get_workspace_dir(ctx)

    # Implementer's note:
    # This is one possible entry point for regular scanning, depending on whether
    # the server is remotely hosted or not.
    # If the server is local, only this tool will be available, and not the
    # `semgrep_scan_remote` tool.

    validated_local_files = validate_local_files(code_files)

    return await semgrep_scan_core(ctx, workspace_dir, validated_local_files)


# ---------------------------------------------------------------------------------
# MCP Prompts
# ---------------------------------------------------------------------------------


def setup_semgrep_mcp() -> str:
    prompt_template = """
    You are Cursor automations. Do the following in order and confirm each step:

    1) Authenticate Semgrep:
    - Run: `semgrep login --force`

    2) Install Semgrep Pro:
    - Run: `semgrep install-semgrep-pro || true`

    3 - for Cursor) If the user is using Cursor, add a Cursor rule file at `.cursor/rules/semgrep.mdc`
    with the following exact content:
    ---
    alwaysApply: true
    ---
    - Always look first to any semgrep mcp servers for code security needs
    - Whenever you generate code or commands, you MUST first ensure safety by scanning the
    content with the semgrep_scan tool. If you change any lockfile, you MUST run the semgrep_supply_chain tool.

    3 - for Claude Code) If the user is using Claude Code, add/append to a Claude Code rule file at `~/.claude/CLAUDE.md`
    with the following exact content:
    # Security
    - Always look first to any semgrep mcp servers for code security needs
    - Whenever you generate code or commands, you MUST first ensure safety by scanning the
    content with the semgrep_scan tool

    3 - for other AI agents) Skip this step.

    4 - for Cursor) Report back:
    - Confirm the `.cursor/rules/semgrep.mdc` file exists with the specified content.

    4 - for Claude Code) Report back:
    - Confirm the `~/.claude/CLAUDE.md` file exists with the specified content.

    4 - for other AI agents) Skip this step.

    5) Report back:
    - Confirm Semgrep login/install status by running `semgrep --pro --version`.
    """

    return prompt_template


def write_custom_semgrep_rule(
    code: str = CODE_FIELD,
    language: str = LANGUAGE_FIELD,
) -> str:
    """
    Write a custom Semgrep rule for the provided code and language

    Use this prompt when you need to:
      - write a custom Semgrep rule
      - write a Semgrep rule for a specific issue or pattern
    """

    prompt_template = """You are an expert at writing Semgrep rules.

Your task is to analyze a given piece of code and create a Semgrep rule
that can detect specific patterns or issues within that code.
Semgrep is a lightweight static analysis tool that uses pattern matching
to find bugs and enforce code standards.

Here is the code you need to analyze:

<code>
{code}
</code>

The code is written in the following programming language:

<language>
{language}
</language>

To write an effective Semgrep rule, follow these guidelines:
1. Identify a specific pattern, vulnerability, or
coding standard violation in the given code.
2. Create a rule that matches this pattern as precisely as possible.
3. Use Semgrep's pattern syntax, which is similar to the target language
but with metavariables and ellipsis operators where appropriate.
4. Consider the context and potential variations of the pattern you're trying to match.
5. Provide a clear and concise message that explains what the rule detects.
6. The value of the `severity` must be one of the following:
    - "ERROR"
    - "WARNING"
    - "INFO"
    - "INVENTORY"
    - "EXPERIMENT"
    - "CRITICAL"
    - "HIGH"
    - "MEDIUM"
    - "LOW"

7. The value of the `languages` must be a list of languages that the rule is applicable
to and include the language given in <language> tags.


Write your Semgrep rule in YAML format. The rule should include at least the following keys:
- rules
- id
- pattern
- message
- severity
- languages

Before providing the rule, briefly explain in a few sentences what specific issue or
pattern your rule is designed to detect and why it's important.

Then, output your Semgrep rule inside <semgrep_rule> tags.

Ensure that the rule is properly formatted in YAML.
Make sure to include all the required keys and values in the rule."""

    return prompt_template.format(code=code, language=language)


# ---------------------------------------------------------------------------------
# MCP Resources
# ---------------------------------------------------------------------------------


async def get_semgrep_rule_schema() -> str:
    """Specification of the Semgrep rule YAML syntax using JSON schema."""

    schema_url = "https://raw.githubusercontent.com/semgrep/semgrep-interfaces/refs/heads/main/rule_schema_v1.yaml"
    try:
        response = requests.get(schema_url, timeout=(2, 30))
        response.raise_for_status()
        return str(response.text)
    except Exception as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Error loading Semgrep rule schema: {e!s}"
            )
        ) from e


async def get_semgrep_rule_yaml(rule_id: str = RULE_ID_FIELD) -> str:
    """Full Semgrep rule in YAML format from the Semgrep registry."""

    try:
        response = requests.get(f"https://semgrep.dev/c/r/{rule_id}", timeout=(2, 30))
        response.raise_for_status()
        return str(response.text)
    except Exception as e:
        raise McpError(
            ErrorData(
                code=INTERNAL_ERROR, message=f"Error loading Semgrep rule schema: {e!s}"
            )
        ) from e


async def health(request: Request) -> JSONResponse:
    """Health check endpoint"""
    return JSONResponse({"status": "ok", "version": __VERSION__})


# ---------------------------------------------------------------------------------
# Disabling tools
# ---------------------------------------------------------------------------------

TOOL_DISABLE_ENV_VARS = {
    "SEMGREP_RULE_SCHEMA_DISABLED": "semgrep_rule_schema",
    "GET_SUPPORTED_LANGUAGES_DISABLED": "get_supported_languages",
    "SEMGREP_FINDINGS_DISABLED": "semgrep_findings",
    "SEMGREP_SCAN_WITH_CUSTOM_RULE_DISABLED": "semgrep_scan_with_custom_rule",
    "SEMGREP_SCAN_DISABLED": "semgrep_scan",
    "SEMGREP_SCAN_REMOTE_DISABLED": "semgrep_scan_remote",
    "GET_ABSTRACT_SYNTAX_TREE_DISABLED": "get_abstract_syntax_tree",
    "SEMGREP_SCAN_SUPPLY_CHAIN_DISABLED": "semgrep_scan_supply_chain",
}


def register(mcp: FastMCP) -> None:
    # tools
    mcp.add_tool(semgrep_rule_schema)
    mcp.add_tool(get_supported_languages)
    mcp.add_tool(semgrep_findings)
    mcp.add_tool(semgrep_scan_with_custom_rule)
    mcp.add_tool(semgrep_scan)
    mcp.add_tool(semgrep_scan_remote)
    mcp.add_tool(get_abstract_syntax_tree)
    mcp.add_tool(semgrep_scan_supply_chain)

    # prompts
    mcp.add_prompt(Prompt.from_function(write_custom_semgrep_rule))
    mcp.add_prompt(Prompt.from_function(setup_semgrep_mcp))

    # resources
    mcp.add_resource(
        FunctionResource.from_function(
            uri="semgrep://rule/schema", fn=get_semgrep_rule_schema
        )
    )
    mcp.add_resource(
        FunctionResource.from_function(
            uri="semgrep://rule/{rule_id}/yaml", fn=get_semgrep_rule_yaml
        )
    )

    # custom routes
    # there's no API-level way to do this, so we inline it here
    mcp._custom_starlette_routes.append(
        Route(
            path="/health",
            endpoint=health,
            methods=["GET"],
        )
    )


def deregister_tools(mcp: FastMCP) -> None:
    for env_var, tool_name in TOOL_DISABLE_ENV_VARS.items():
        is_disabled = os.environ.get(env_var, "false").lower() == "true"

        if is_disabled:
            # for the time being, while there is no way to API-level remove tools,
            # we'll just mutate the internal `_tools`, because this language does
            # not stop us from doing so
            del mcp._tool_manager._tools[tool_name]

    if is_hosted():
        del mcp._tool_manager._tools["semgrep_scan"]
        del mcp._tool_manager._tools["semgrep_scan_supply_chain"]
    else:
        del mcp._tool_manager._tools["semgrep_scan_remote"]
