import { ApiSecurity, ApiUrl, AppDetails } from '../shared';
import { TeamsSettings } from '../shared/types/teams';
import { UserSettings, UUID } from '../shared/types/userSettings';
import { ChangePasswordWithLinkPayload, CryptoProvider, Keys, LoginDetails, PrivateKeys, RegisterDetails, RegisterOpaqueDetails, RegisterPreCreatedUser, RegisterPreCreatedUserResponse, SecurityDetails, Token, TwoFactorAuthQR } from './types';
import { paths } from '../schema';
export * from './types';
export declare class Auth {
    private readonly client;
    private readonly appDetails;
    private readonly apiSecurity?;
    private readonly apiUrl;
    static client(apiUrl: ApiUrl, appDetails: AppDetails, apiSecurity?: ApiSecurity): Auth;
    private constructor();
    /**
     * Runs the first phase (out of 2) of opaque registation of a new user
     * @param email - The user email.
     * @param registrationRequest - The opaque registration request.
     * @returns The opaque sign up response.
     */
    registerOpaqueStart(email: string, registrationRequest: string): Promise<{
        signUpResponse: string;
    }>;
    /**
     * Runs the second phase (out of 2) of opaque registation of a new user
     * @param registrationRecord - The opaque registration record.
     * @param registerDetails - The user registration details.
     * @param startLoginRequest - The opaque start login request.
     * @returns The opaque login response.
     */
    registerOpaqueFinish(registrationRecord: string, registerDetails: RegisterOpaqueDetails, startLoginRequest: string): Promise<{
        loginResponse: string;
    }>;
    /**
     * Tries to register a new user
     * @param registerDetails
     */
    register(registerDetails: RegisterDetails): Promise<{
        token: Token;
        newToken: Token;
        user: Omit<UserSettings, 'bucket'> & {
            referralCode: string;
        };
        uuid: UUID;
    }>;
    /**
     * Tries to register a new user without sending keys
     * @param registerDetails
     */
    registerWithoutKeys(registerDetails: Omit<RegisterDetails, 'keys'>): Promise<{
        token: Token;
        newToken: Token;
        user: Omit<UserSettings, 'bucket'> & {
            referralCode: string;
        };
        uuid: UUID;
    }>;
    /**
     * Registers a precreated user
     * @param registerDetails
     * @returns {Promise<RegisterPreCreatedUserResponse>} Returns sign in token, user data and uuid.
     */
    registerPreCreatedUser(registerDetails: RegisterPreCreatedUser): Promise<RegisterPreCreatedUserResponse>;
    /**
     * Registers a precreated user without sending keys
     * @param registerDetails
     * @returns {Promise<RegisterPreCreatedUserResponse>} Returns sign in token, user data and uuid.
     */
    registerPreCreatedUserWithoutKeys(registerDetails: RegisterPreCreatedUser): Promise<RegisterPreCreatedUserResponse>;
    /**
     * Requests account unblock email
     *
     * @param {string} email - The email address associated with the account.
     * @returns {Promise<void>} - Resolves when the email is sent.
     */
    requestUnblockAccount(email: string): Promise<void>;
    /**
     * Unblocks account with token
     *
     * @param {string} token - The token received via email to verify and unblock the account.
     * @returns {Promise<void>} - Resolves successfuly when account is unblocked
     */
    unblockAccount(token: string): Promise<void>;
    /**
     * Runs the first phase (out of 2) of opaque log for the given user
     * @param email - The user email.
     * @param startLoginRequest - The opaque start login request.
     * @param tfa - The two factor auth code.
     * @returns The opaque login response.
     */
    loginOpaqueStart(email: string, startLoginRequest: string): Promise<{
        loginResponse: string;
    }>;
    /**
     * Runs the second phase (out of 2) of opaque log for the given user
     * @param email
     * @param finishLoginRequest
     */
    loginOpaqueFinish(email: string, finishLoginRequest: string, tfa: string | undefined): Promise<{
        sessionID: string;
        user: UserSettings;
        token: string;
    }>;
    /**
     * Tries to log in a user given its login details
     * @param details
     * @param cryptoProvider
     */
    login(details: LoginDetails, cryptoProvider: CryptoProvider): Promise<{
        token: Token;
        newToken: Token;
        user: UserSettings;
        userTeam: TeamsSettings | null;
    }>;
    /**
     * Tries to log in a user given its cli login details
     * @param details
     * @param cryptoProvider
     */
    loginAccess(details: LoginDetails, cryptoProvider: CryptoProvider): Promise<paths['/auth/cli/login/access']['post']['responses']['200']['content']['application/json']>;
    /**
     * Tries to log in a user given its login details without sending keys
     * @param details
     * @param cryptoProvider
     */
    loginWithoutKeys(details: LoginDetails, cryptoProvider: CryptoProvider): Promise<{
        token: Token;
        newToken: Token;
        user: UserSettings;
        userTeam: TeamsSettings | null;
    }>;
    /**
     * Updates the asymmetric keys
     * @param keys
     * @param token
     */
    updateKeys(keys: Keys, token: Token): Promise<unknown>;
    /**
     * Returns general security details
     * @param email
     */
    securityDetails(email: string): Promise<SecurityDetails>;
    /**
     * Logout
     */
    logout(token?: Token): Promise<void>;
    /**
     * Generates a new TwoFactorAuth code
     */
    generateTwoFactorAuthQR(token?: Token): Promise<TwoFactorAuthQR>;
    /**
     * Disables TwoFactorAuthentication
     * @param pass
     * @param code
     */
    disableTwoFactorAuth(pass: string, code: string, token?: Token): Promise<void>;
    /**
     * Store TwoFactorAuthentication details
     * @param backupKey
     * @param code
     */
    storeTwoFactorAuthKey(backupKey: string, code: string, token?: Token): Promise<void>;
    /**
     * @deprecated Use `sendUserDeactivationEmail` instead.
     * Sends request to send the email to delete the account
     * @param email
     */
    sendDeactivationEmail(email: string): Promise<void>;
    /**
     * Sends request to send the email to delete the account
     */
    sendUserDeactivationEmail(token?: Token): Promise<void>;
    /**
     * @deprecated Use `confirmUserDeactivation` instead.
     * Confirms the account deactivation
     * @param token
     */
    confirmDeactivation(token: string): Promise<void>;
    /**
     * Confirms the account deactivation
     * @param token
     */
    confirmUserDeactivation(deactivationToken: string, token?: string): Promise<void>;
    /**
     * Check credentials
     * @param hashedPassword
     * @returns
     */
    areCredentialsCorrect(hashedPassword: string, token?: Token): Promise<boolean>;
    /**
     * Send email to change password
     * @param email
     */
    sendChangePasswordEmail(email: string): Promise<void>;
    /**
     * Restore password with email link
     * @param token
     * @param password
     * @param salt
     * @param mnemonic
     * @param keys
     */
    changePasswordWithLink(token: string | undefined, password: string, salt: string, mnemonic: string, keys?: PrivateKeys): Promise<void>;
    /**
     * Restore password with email link
     * @param token
     * @param password
     * @param salt
     * @param mnemonic
     * @param keys
     */
    changePasswordWithLinkV2(token: string | undefined, password: string, salt: string, mnemonic: string, keys?: PrivateKeys): Promise<void>;
    legacyRecoverAccount({ token, encryptedPassword, encryptedSalt, encryptedMnemonic, eccEncryptedMnemonic, kyberEncryptedMnemonic, keys, }: ChangePasswordWithLinkPayload): Promise<void>;
    /**
     * Reset account with token
     * @param token
     * @param password
     * @param salt
     * @param mnemonic
     */
    resetAccountWithToken(token: string | undefined, password: string, salt: string, mnemonic: string): Promise<void>;
    private basicHeaders;
    private headersWithToken;
}
