import { Token } from '../auth';
import { ListAllSharedFoldersResponse, SharingMeta } from '../drive/share/types';
import { CreateFolderResponse, DriveFileData, FetchPaginatedFolderContentResponse, FetchTrashContentResponse } from '../drive/storage/types';
import { ApiSecurity, ApiUrl, AppDetails } from '../shared';
import { RequestCanceler } from '../shared/http/client';
import { CreateFolderPayload, CreateTeamData, CreateWorkspaceSharingPayload, FileEntry, GetMemberDetailsResponse, GetMemberUsageResponse, InviteMemberBody, ListWorkspaceSharedItemsResponse, OrderByOptions, PendingInvitesResponse, Workspace, WorkspaceCredentialsDetails, WorkspaceMembers, WorkspacePendingInvitations, WorkspaceSetupInfo, WorkspacesResponse, WorkspaceTeamResponse, TeamMembers, UsersAndTeamsAnItemIsShareWidthResponse, WorkspaceUser, WorkspaceUsage, ItemType, WorkspaceLogOrderBy, WorkspaceLogType, WorkspaceLogResponse } from './types';
export declare class Workspaces {
    private readonly client;
    private readonly appDetails;
    private readonly apiSecurity;
    static client(apiUrl: ApiUrl, appDetails: AppDetails, apiSecurity: ApiSecurity): Workspaces;
    private constructor();
    /**
     * Returns the needed headers for the module requests
     * @private
     */
    private headers;
    private getRequestHeaders;
    getWorkspaces(): Promise<WorkspacesResponse>;
    getPendingWorkspaces(): Promise<WorkspacesResponse>;
    getPendingInvites(): Promise<PendingInvitesResponse>;
    validateWorkspaceInvite(inviteId: string): Promise<string>;
    /**
     * Uploads an avatar for a specific workspace.
     * @param workspaceId The UUID of the workspace to upload the avatar for.
     * @param avatar The avatar to upload.
     * @returns The response from the server.
     */
    uploadWorkspaceAvatar(workspaceId: string, avatar: Blob): Promise<void>;
    deleteWorkspaceAvatar(workspaceId: string): Promise<void>;
    setupWorkspace({ workspaceId, name, address, description, encryptedMnemonic, }: WorkspaceSetupInfo): Promise<void>;
    getWorkspaceUsage(workspaceId: string): Promise<WorkspaceUsage>;
    editWorkspace(workspaceId: string, details: {
        name?: string;
        description?: string;
        address?: string;
    }): Promise<void>;
    updateAvatar(workspaceId: string, payload: {
        avatar: Blob;
    }): Promise<{
        avatar: string;
    }>;
    deleteAvatar(workspaceId: string): Promise<void>;
    createTeam({ workspaceId, name, managerId }: CreateTeamData): Promise<void>;
    getWorkspacesMembers(workspaceId: string): Promise<WorkspaceMembers>;
    getWorkspacesTeams(workspaceId: string): Promise<WorkspaceTeamResponse>;
    editTeam({ teamId, name }: {
        teamId: string;
        name: string;
    }): Promise<void>;
    deleteTeam({ workspaceId, teamId }: {
        workspaceId: string;
        teamId: string;
    }): Promise<void>;
    getWorkspacesTeamMembers(teamId: string): Promise<TeamMembers>;
    addTeamUser(teamId: string, userUuid: string): Promise<void>;
    removeTeamUser(teamId: string, userUuid: string): Promise<void>;
    changeTeamManager(workspaceId: string, teamId: string, userUuid: string): Promise<void>;
    getPersonalTrash(workspaceId: string, type: ItemType, offset?: number, limit?: number): Promise<FetchTrashContentResponse>;
    emptyPersonalTrash(workspaceId: string): Promise<void>;
    changeUserRole(teamId: string, memberId: string, role: string): Promise<void>;
    inviteMemberToWorkspace({ workspaceId, invitedUserEmail, spaceLimitBytes, encryptedMnemonicInBase64, encryptionAlgorithm, message, }: InviteMemberBody): Promise<void>;
    leaveWorkspace(workspaceId: string): Promise<void>;
    getMemberDetails(workspaceId: string, memberId: string): Promise<GetMemberDetailsResponse>;
    modifyMemberUsage(workspaceId: string, memberId: string, spaceLimitBytes: number): Promise<WorkspaceUser>;
    getMemberUsage(workspaceId: string): Promise<GetMemberUsageResponse>;
    deactivateMember(workspaceId: string, memberId: string): Promise<void>;
    activateMember(workspaceId: string, memberId: string): Promise<void>;
    removeMember(workspaceId: string, memberId: string): Promise<void>;
    acceptInvitation(inviteId: string, token: string): Promise<void>;
    declineInvitation(inviteId: string, token: string): Promise<void>;
    getWorkspaceCredentials(workspaceId: string): Promise<WorkspaceCredentialsDetails>;
    createFileEntry(fileEntry: FileEntry, workspaceId: string, resourcesToken?: Token): Promise<DriveFileData>;
    /**
     * Creates a new folder in the specified workspace.
     *
     * @param {CreateFolderPayload} options - The options for creating the folder.
     * @param {string} options.workspaceId - The ID of the workspace.
     * @param {string} options.plainName - The plain name of the folder.
     * @param {string} options.parentFolderUuid - The UUID of the parent folder.
     * @return {[Promise<CreateFolderResponse>, RequestCanceler]} A tuple containing a promise to get the API response and a function to cancel the request.
     */
    createFolder({ workspaceId, plainName, parentFolderUuid, }: CreateFolderPayload): [Promise<CreateFolderResponse>, RequestCanceler];
    /**
     * Retrieves a paginated list of folders within a specific folder in a workspace.
     *
     * @param {string} workspaceId - The ID of the workspace.
     * @param {string} folderUUID - The UUID of the folder.
     * @param {number} [offset=0] - The position of the first file to return.
     * @param {number} [limit=50] - The max number of files to be returned.
     * @param {string} [sort=''] - The reference column to sort it.
     * @param {string} [order=''] - The order to be followed.
     * @return {[Promise<FetchPaginatedFolderContentResponse>, RequestCanceler]} An array containing a promise to get the API response and a function to cancel the request.
     */
    getFolders(workspaceId: string, folderUUID: string, offset?: number, limit?: number, sort?: string, order?: string): [Promise<FetchPaginatedFolderContentResponse>, RequestCanceler];
    /**
     * Retrieves a paginated list of files within a specific folder in a workspace.
     *
     * @param {string} workspaceId - The ID of the workspace.
     * @param {string} folderUUID - The UUID of the folder.
     * @param {number} [offset=0] - The position of the first file to return.
     * @param {number} [limit=50] - The max number of files to be returned.
     * @param {string} [sort=''] - The reference column to sort it.
     * @param {string} [order=''] - The order to be followed.
     * @return {[Promise<FetchPaginatedFolderContentResponse>, RequestCanceler]} An array containing a promise to get the API response and a function to cancel the request.
     */
    getFiles(workspaceId: string, folderUUID: string, offset?: number, limit?: number, sort?: string, order?: string): [Promise<FetchPaginatedFolderContentResponse>, RequestCanceler];
    getWorkspacePendingInvitations(workspaceId: string, limit: number, offset: number): Promise<WorkspacePendingInvitations[]>;
    /**
     * Creates a new sharing for a workspace item.
     *
     * @param {CreateWorkspaceSharingPayload} options - The options for creating the sharing.
     * @param {string} options.workspaceId - The ID of the workspace.
     * @param {string} options.teamUUID - The UUID of the team.
     * @param {...CreateSharingPayload} options.createSharingPayload - The payload for creating the sharing.
     * @returns {Promise<SharingMeta>} A promise that resolves to the sharing metadata.
     */
    shareItem({ workspaceId, teamUUID, ...createSharingPayload }: CreateWorkspaceSharingPayload): Promise<SharingMeta>;
    validateWorkspaceInvitation(inviteId: string): Promise<{
        uuid: string;
    }>;
    /**
     * Returns shared files in teams.
     * @param orderBy
     * @deprecated use `getWorkspaceTeamSharedFilesV2` call instead.
     */
    getWorkspaceTeamSharedFiles(workspaceId: string, teamId: string, orderBy?: OrderByOptions): [Promise<ListAllSharedFoldersResponse>, RequestCanceler];
    /**
     * Returns shared files in teams.
     * @param orderBy
     */
    getWorkspaceTeamSharedFilesV2(workspaceId: string, orderBy?: OrderByOptions): [Promise<ListAllSharedFoldersResponse>, RequestCanceler];
    /**
     * Returns shared folders in teams.
     * @param orderBy
     * @deprecated use `getWorkspaceTeamSharedFoldersV2` call instead.
     */
    getWorkspaceTeamSharedFolders(workspaceId: string, teamId: string, orderBy?: OrderByOptions): [Promise<ListAllSharedFoldersResponse>, RequestCanceler];
    /**
     * Returns shared folders in teams.
     * @param orderBy
     */
    getWorkspaceTeamSharedFoldersV2(workspaceId: string, orderBy?: OrderByOptions): [Promise<ListAllSharedFoldersResponse>, RequestCanceler];
    /**
     * Returns files inside a shared folders with teams.
     * @param orderBy
     * @deprecated use `getWorkspaceTeamSharedFolderFilesV2` call instead.
     */
    getWorkspaceTeamSharedFolderFiles(workspaceId: string, teamId: string, sharedFolderUUID: string, page?: number, perPage?: number, token?: string, orderBy?: OrderByOptions): [Promise<ListWorkspaceSharedItemsResponse>, RequestCanceler];
    /**
     * Returns files inside a shared folders with teams.
     * @param orderBy
     */
    getWorkspaceTeamSharedFolderFilesV2(workspaceId: string, sharedFolderUUID: string, page?: number, perPage?: number, token?: string, orderBy?: OrderByOptions): [Promise<ListWorkspaceSharedItemsResponse>, RequestCanceler];
    /**
     * Returns folders inside a shared folders with teams.
     * @param orderBy
     * @deprecated use `getWorkspaceTeamSharedFolderFoldersV2` call instead.
     */
    getWorkspaceTeamSharedFolderFolders(workspaceId: string, teamId: string, sharedFolderUUID: string, page?: number, perPage?: number, token?: string, orderBy?: OrderByOptions): [Promise<ListWorkspaceSharedItemsResponse>, RequestCanceler];
    /**
     * Returns folders inside a shared folders with teams.
     * @param orderBy
     */
    getWorkspaceTeamSharedFolderFoldersV2(workspaceId: string, sharedFolderUUID: string, page?: number, perPage?: number, token?: string, orderBy?: OrderByOptions): [Promise<ListWorkspaceSharedItemsResponse>, RequestCanceler];
    /**
     * Returns users and teams an item is shared with.
     */
    getUsersAndTeamsAnItemIsShareWidth({ workspaceId, itemType, itemId, }: {
        workspaceId: string;
        itemType: ItemType;
        itemId: string;
    }): [Promise<UsersAndTeamsAnItemIsShareWidthResponse>, RequestCanceler];
    getWorkspace(workspaceId: string): Promise<Workspace>;
    getWorkspaceLogs(workspaceId: string, limit?: number, offset?: number, member?: string, activity?: WorkspaceLogType[], lastDays?: number, summary?: boolean, orderBy?: WorkspaceLogOrderBy): Promise<WorkspaceLogResponse[]>;
}
export * from './types';
