import { Token } from '../../auth';
import { ApiSecurity, ApiUrl, AppDetails } from '../../shared';
import { RequestCanceler } from '../../shared/http/client';
import { UUID } from '../../shared/types/userSettings';
import { ItemType } from './../../workspaces/types';
import { AddItemsToTrashPayload, CheckDuplicatedFilesPayload, CheckDuplicatedFilesResponse, CheckDuplicatedFolderPayload, CheckDuplicatedFoldersResponse, CreateFolderByUuidPayload, CreateFolderPayload, CreateFolderResponse, CreateThumbnailEntryPayload, DeleteFilePayload, DriveFileData, FetchFolderContentResponse, FetchLimitResponse, FetchPaginatedFilesContent, FetchPaginatedFolderContentResponse, FetchPaginatedFoldersContent, FileEntry, FileEntryByUuid, FileMeta, FolderAncestor, FolderAncestorWorkspace, FolderMeta, FolderTreeResponse, MoveFilePayload, MoveFileResponse, MoveFileUuidPayload, MoveFolderPayload, MoveFolderResponse, MoveFolderUuidPayload, ReplaceFile, SearchResultData, Thumbnail, ThumbnailEntry, UpdateFilePayload, UpdateFolderMetadataPayload, UsageResponse, UsageResponseV2 } from './types';
export * as StorageTypes from './types';
export declare class Storage {
    private readonly client;
    private readonly appDetails;
    private readonly apiSecurity;
    static client(apiUrl: ApiUrl, appDetails: AppDetails, apiSecurity: ApiSecurity): Storage;
    private constructor();
    /**
     * Creates a new folder
     * @param payload
     */
    createFolder(payload: CreateFolderPayload): [Promise<CreateFolderResponse>, RequestCanceler];
    /**
     * Creates a new folder
     * @param payload
     */
    createFolderByUuid(payload: CreateFolderByUuidPayload): [Promise<CreateFolderResponse>, RequestCanceler];
    /**
     * Moves a specific folder to a new location
     * @param payload
     * @deprecated Use `moveFolderByUuid` instead.
     */
    moveFolder(payload: MoveFolderPayload): Promise<MoveFolderResponse>;
    /**
     * Moves a specific folder to a new location
     * @param payload
     */
    moveFolderByUuid(uuid: string, payload: MoveFolderUuidPayload): Promise<FolderMeta>;
    /**
     * Updates the metadata of a folder
     * @param payload
     */
    updateFolder(payload: UpdateFolderMetadataPayload): Promise<void>;
    /**
     * Updates the name of a folder with the given UUID.
     *
     * @param {Object} payload - The payload containing the folder UUID and the new name.
     * @param {string} payload.folderUuid - The UUID of the folder to update.
     * @param {string} payload.name - The new name for the folder.
     * @param {Token} [resourcesToken] - An optional token for authentication.
     * @return {Promise<void>} A promise that resolves when the folder name is successfully updated.
     */
    updateFolderNameWithUUID(payload: {
        folderUuid: string;
        name: string;
    }, resourcesToken?: Token): Promise<void>;
    /**
     * Fetches & returns the contents of a specific folder
     * @param folderId
     */
    getFolderContent(folderId: number, trash?: boolean): [Promise<FetchFolderContentResponse>, RequestCanceler];
    /**
     * Fetches and returns the contents of a specific folder by its UUID.
     *
     * @param {string} folderUuid - The UUID of the folder.
     * @param {boolean} [trash=false] - Whether to include trash items in the response.
     * @param {boolean} [offset] - The position of the first file to return.
     * @param {boolean} [limit] - The max number of files to be returned.
     * @param {boolean} [workspacesToken] - Token for accessing workspaces.
     * @return {[Promise<FetchFolderContentResponse>, RequestCanceler]} An array containing a promise to get the API response and a function to cancel the request.
     */
    getFolderContentByUuid({ folderUuid, trash, offset, limit, workspacesToken, }: {
        folderUuid: string;
        trash?: boolean;
        limit?: number;
        offset?: number;
        workspacesToken?: string;
    }): [Promise<FetchFolderContentResponse>, RequestCanceler];
    /**
     * Retrieves a file with the specified fileId along with the associated workspacesToken.
     *
     * @param {string} fileId - The ID of the file to retrieve.
     * @param {string} [workspacesToken] - Token for accessing workspaces.
     * @return {[Promise<FileMeta>, RequestCanceler]} A promise with FileMeta and a canceler for the request.
     */
    getFile(fileId: string, workspacesToken?: string): [Promise<FileMeta>, RequestCanceler];
    /**
     * Gets the files in a folder.
     *
     * @param {number} folderId - The ID of the folder.
     * @param {number} [offset=0] - The position of the first file to return.
     * @param {number} [limit=50] - The max number of files to be returned.
     * @param {string} [sort=plainName] - The reference column to sort it.
     * @param {string} [order=ASC] - The order to be followed.
     * @returns {[Promise<FetchPaginatedFolderContentResponse>, RequestCanceler]} An array containing a promise to get the API response and a function to cancel the request.
     */
    getFolderFiles(folderId: number, offset?: number, limit?: number, sort?: string, order?: string): [Promise<FetchPaginatedFolderContentResponse>, RequestCanceler];
    /**
     * Gets the files in a folder by its UUID.
     *
     * @param {UUID} folderUuid - The UUID of the folder.
     * @param {number} [offset=0] - The position of the first file to return.
     * @param {number} [limit=50] - The max number of files to be returned.
     * @param {string} [sort=plainName] - The reference column to sort it.
     * @param {string} [order=ASC] - The order to be followed.
     * @returns {[Promise<FetchPaginatedFilesContent>, RequestCanceler]} An array containing a promise to get the API response and a function to cancel the request.
     */
    getFolderFilesByUuid(folderUuid: UUID, offset?: number, limit?: number, sort?: string, order?: string): [Promise<FetchPaginatedFilesContent>, RequestCanceler];
    /**
     * Gets the subfolders of a folder.
     *
     * @param {number} folderId - The ID of the folder.
     * @param {number} [offset=0] - The position of the first subfolder to return.
     * @param {number} [limit=50] - The max number of subfolders to return.
     * @param {string} [sort=plainName] - The reference column to sort it.
     * @param {string} [order=ASC] - The order to be followed.
     * @returns {[Promise<FetchPaginatedFolderContentResponse>, RequestCanceler]} An array containing a promise to get the API response and a function to cancel the request.
     */
    getFolderFolders(folderId: number, offset?: number, limit?: number, sort?: string, order?: string): [Promise<FetchPaginatedFolderContentResponse>, RequestCanceler];
    /**
     * Gets the subfolders of a folder by its UUID.
     *
     * @param {UUID} folderUuid - The UUID of the folder.
     * @param {number} [offset=0] - The position of the first subfolder to return.
     * @param {number} [limit=50] - The max number of subfolders to return.
     * @param {string} [sort=plainName] - The reference column to sort it.
     * @param {string} [order=ASC] - The order to be followed.
     * @returns {[Promise<FetchPaginatedFoldersContent>, RequestCanceler]} An array containing a promise to get the API response and a function to cancel the request.
     */
    getFolderFoldersByUuid(folderUuid: UUID, offset?: number, limit?: number, sort?: string, order?: string): [Promise<FetchPaginatedFoldersContent>, RequestCanceler];
    /**
     * Removes a specific folder from the centralized persistence
     * @param folderId
     * @deprecated Use `deleteFolderByUuid` instead.
     */
    deleteFolder(folderId: number): Promise<unknown>;
    /**
     * Deletes a folder from the storage system using its unique identifier (UUID).
     *
     * @param folderId - The UUID of the folder to be deleted.
     * @returns A promise that resolves with the response of the delete operation.
     */
    deleteFolderByUuid(folderId: string): Promise<void>;
    /**
     * Returns the total size of a folder
     * @param folderId
     */
    getFolderSize(folderId: number): Promise<number>;
    /**
     * Creates a new file entry
     * @param fileEntry
     * @deprecated Use `createFileEntryByUuid` instead.
     */
    createFileEntry(fileEntry: FileEntry, resourcesToken?: Token): Promise<DriveFileData>;
    /**
     * Creates a new file entry
     * @param fileEntry
     */
    createFileEntryByUuid(fileEntry: FileEntryByUuid, resourcesToken?: string): Promise<DriveFileData>;
    /**
     * Creates a new thumbnail entry
     * @param thumbnailEntry
     */
    createThumbnailEntry(thumbnailEntry: ThumbnailEntry, resourcesToken?: Token): Promise<Thumbnail>;
    /**
     * Creates a new thumbnail entry using drive-server-wip
     * @param CreateThumbnailEntryPayload
     */
    createThumbnailEntryWithUUID(thumbnailEntry: CreateThumbnailEntryPayload, resourcesToken?: string): Promise<Thumbnail>;
    /**
     * Updates the details of a file entry
     * @param payload
     */
    updateFile(payload: UpdateFilePayload, resourcesToken?: Token): Promise<void>;
    /**
     * Updates the name of a file with the given UUID.
     *
     * @param {Object} payload - The payload containing the UUID and new name of the file.
     * @param {string} payload.fileUuid - The UUID of the file.
     * @param {string} payload.name - The new name of the file.
     * @param {string} [resourcesToken] - The token for accessing resources.
     * @return {Promise<void>} - A Promise that resolves when the file name is successfully updated.
     */
    updateFileNameWithUUID(payload: {
        fileUuid: string;
        name: string;
    }, resourcesToken?: Token): Promise<void>;
    /**
     * Updates the name and the type from a given file UUID.
     *
     * @param {Object} payload - The payload containing the UUID and the new properties of the file.
     * @param {string} payload.fileUuid - The UUID of the file.
     * @param {string} payload.name - The new name of the file.
     * @param {string} payload.type - The new type of the file.
     * @param {string} [resourcesToken] - The token for accessing resources.
     * @return {Promise<void>} - A Promise that resolves when the file name is successfully updated.
     */
    updateFileMetaByUUID(fileUuid: string, payload: {
        plainName?: string;
        type?: string | null;
    }, resourcesToken?: Token): Promise<void>;
    /**
     * Deletes a specific file entry
     * @param payload
     * @deprecated Use `deleteFileByUuid` instead.
     */
    deleteFile(payload: DeleteFilePayload): Promise<unknown>;
    /**
     * Deletes a file from the storage system using its unique identifier (UUID).
     *
     * @param fileId - The UUID of the file to be deleted.
     * @returns A promise that resolves with the response of the delete operation.
     */
    deleteFileByUuid(fileId: string): Promise<{
        deleted: boolean;
    }>;
    /**
     * Updates the persisted path of a file entry
     * @param payload
     * @deprecated Use `moveFileByUuid` instead.
     */
    moveFile(payload: MoveFilePayload): Promise<MoveFileResponse>;
    /**
     * Moves a specific file to a new location
     * @param payload
     */
    moveFileByUuid(uuid: string, payload: MoveFileUuidPayload): Promise<FileMeta>;
    /**
     * Returns a list of the n most recent files
     * @param limit
     * @deprecated use `getRecentFilesV2` call instead.
     */
    getRecentFiles(limit: number): Promise<DriveFileData[]>;
    /**
     * Returns a list of the n most recent files
     * @param limit
     */
    getRecentFilesV2(limit: number): Promise<DriveFileData[]>;
    /**
     * Returns a list of items in trash
     */
    getTrash(): [Promise<FetchFolderContentResponse>, RequestCanceler];
    /**
     * Add Items to Trash
     * @param payload
     */
    addItemsToTrash(payload: AddItemsToTrashPayload): Promise<void>;
    /**
     * @returns whether the user has uploaded any files
     */
    hasUploadedFiles(): Promise<{
        hasUploadedFiles: boolean;
    }>;
    /**
     * Returns a list of the n most recent files
     * @param limit
     */
    searchItemsByName(plain_name: string): Promise<DriveFileData[]>;
    /**
     * Returns the current space usage of the user
     * @deprecated use `spaceUsageV2` call instead.
     */
    spaceUsage(): Promise<UsageResponse>;
    /**
     * Returns the current space usage of the user
     */
    spaceUsageV2(): Promise<UsageResponseV2>;
    /**
     * Returns the current space limit for the user
     * @deprecated use `spaceLimitV2` call instead.
     */
    spaceLimit(): Promise<FetchLimitResponse>;
    /**
     * Returns the current space limit for the user
     */
    spaceLimitV2(): Promise<FetchLimitResponse>;
    /**
     * Get global search items.
     *
     * @param {string} search - The name of the item.
     * @param {string} workspaceId - The ID of the workspace (optional).
     * @param {number} offset - The position of the first item to return (optional).
     * @returns {[Promise<SearchResultData>, RequestCanceler]} An array containing a promise to get the API response and a function to cancel the request.
     */
    getGlobalSearchItems(search: string, workspaceId?: string, offset?: number): [Promise<SearchResultData>, RequestCanceler];
    /**
     * Returns the needed headers for the module requests
     * @private
     */
    private headers;
    /**
     * Gets the ancestors of a given folder UUID
     *
     * @param {string} uuid - UUID of the folder.
     * @returns {Promise<FolderAncestor[]>} A promise that resolves with an array of ancestors of the given folder.
     */
    getFolderAncestors(uuid: string): Promise<FolderAncestor[]>;
    /**
     * Gets the ancestors of an item with the given UUID and type in a Workspace
     *
     * @param {string} workspaceId - UUID of the workspace.
     * @param {string} itemType - itemType to know if the item is file or folder
     * @param {string} uuid - UUID of the item.
     * @returns {Promise<FolderAncestor[]>} A promise that resolves with an array of ancestors of the given folder.
     */
    getFolderAncestorsInWorkspace(workspaceId: string, itemType: ItemType, uuid: string, resourcesToken?: Token): Promise<FolderAncestorWorkspace[]>;
    /**
     * Gets the meta of a given folder UUID
     *
     * @param {string} folderUUID - UUID of the folder.
     * @returns {Promise<FolderMeta>}
     */
    getFolderMeta(uuid: string, workspacesToken?: string, resourcesToken?: string): Promise<FolderMeta>;
    /**
     * Gets the meta of a given folder Id
     *
     * @param {number} folderId - Id of the folder.
     * @returns {Promise<FolderMeta>}
     */
    getFolderMetaById(folderId: number): Promise<FolderMeta>;
    /**
     * Replaces a file with a new one.
     *
     * @param {string} uuid - UUID of the file.
     * @param {ReplaceFile} payload
     * @returns {Promise<DriveFileData>} - The replaced file data.
     */
    replaceFile(uuid: string, payload: ReplaceFile): Promise<DriveFileData>;
    /**
     * Checks the size limit for a file.
     *
     * @param {number} size - The size of the file to check.
     * @return {Promise<void>} - A promise that resolves when the size limit check is complete.
     */
    checkSizeLimit(size: number): Promise<void>;
    /**
     * Retrieves the folder tree based on the UUID.
     *
     * @param {string} uuid - The UUID of the folder.
     * @return {Promise<FolderTreeResponse>} The promise containing the folder tree response.
     */
    getFolderTree(uuid: string): Promise<FolderTreeResponse>;
    /**
     * Checks if the given files already exist in the given folder.
     *
     * @param {CheckDuplicatedFilesPayload} payload - Payload containing the folder UUID and the list of files to check.
     * @return {Promise<CheckDuplicatedFilesResponse>} - Promise that contains the duplicated files in a list.
     */
    checkDuplicatedFiles({ folderUuid, filesList, }: CheckDuplicatedFilesPayload): Promise<CheckDuplicatedFilesResponse>;
    /**
     * Checks if the given folders names already exist in the given folder
     *
     * @param {CheckDuplicatedFolderPayload} payload - Payload containing the folder UUID and the list of folders to check.
     * @return {Promise<CheckDuplicatedFoldersResponse>} - Promise that contains the duplicated folders in a list.
     */
    checkDuplicatedFolders({ folderUuid, folderNamesList, }: CheckDuplicatedFolderPayload): Promise<CheckDuplicatedFoldersResponse>;
    /**
     * Gets the folder meta from a given path (e.g. "/folder1/folder2")
     *
     * @param {string} folderPath - The path of the folder.
     * @returns {Promise<FolderMeta>} A promise that resolves the folder on that path.
     */
    getFolderByPath(folderPath: string): Promise<FolderMeta>;
    /**
     * Gets the file meta from a given path (e.g. "/folder1/folder2/file.png")
     *
     * @param {string} filePath - The path of the file.
     * @returns {Promise<FileMeta>} A promise that resolves the file on that path.
     */
    getFileByPath(filePath: string): Promise<FileMeta>;
}
