"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebDavServer = void 0;
const https_1 = __importDefault(require("https"));
const http_1 = __importDefault(require("http"));
const config_service_1 = require("../services/config.service");
const OPTIONS_handler_1 = require("./handlers/OPTIONS.handler");
const PROPFIND_handler_1 = require("./handlers/PROPFIND.handler");
const logger_utils_1 = require("../utils/logger.utils");
const body_parser_1 = __importDefault(require("body-parser"));
const auth_middleware_1 = require("./middewares/auth.middleware");
const request_logger_middleware_1 = require("./middewares/request-logger.middleware");
const GET_handler_1 = require("./handlers/GET.handler");
const HEAD_handler_1 = require("./handlers/HEAD.handler");
const download_service_1 = require("../services/network/download.service");
const auth_service_1 = require("../services/auth.service");
const crypto_service_1 = require("../services/crypto.service");
const errors_middleware_1 = require("./middewares/errors.middleware");
const express_async_handler_1 = __importDefault(require("express-async-handler"));
const sdk_manager_service_1 = require("../services/sdk-manager.service");
const network_facade_service_1 = require("../services/network/network-facade.service");
const network_utils_1 = require("../utils/network.utils");
const PUT_handler_1 = require("./handlers/PUT.handler");
const MKCOL_handler_1 = require("./handlers/MKCOL.handler");
const DELETE_handler_1 = require("./handlers/DELETE.handler");
const PROPPATCH_handler_1 = require("./handlers/PROPPATCH.handler");
const MOVE_handler_1 = require("./handlers/MOVE.handler");
const COPY_handler_1 = require("./handlers/COPY.handler");
const inxt_js_1 = require("@internxt/inxt-js");
const mkcol_middleware_1 = require("./middewares/mkcol.middleware");
const webdav_folder_service_1 = require("./services/webdav-folder.service");
class WebDavServer {
    app;
    configService;
    driveFileService;
    driveFolderService;
    downloadService;
    authService;
    cryptoService;
    trashService;
    constructor(app, configService, driveFileService, driveFolderService, downloadService, authService, cryptoService, trashService) {
        this.app = app;
        this.configService = configService;
        this.driveFileService = driveFileService;
        this.driveFolderService = driveFolderService;
        this.downloadService = downloadService;
        this.authService = authService;
        this.cryptoService = cryptoService;
        this.trashService = trashService;
    }
    getNetworkFacade = async () => {
        const credentials = await this.configService.readUser();
        if (!credentials)
            throw new Error('Credentials not found in Config service, do login first');
        const networkModule = sdk_manager_service_1.SdkManager.instance.getNetwork({
            user: credentials.user.bridgeUser,
            pass: credentials.user.userId,
        });
        const environment = new inxt_js_1.Environment({
            bridgeUser: credentials.user.bridgeUser,
            bridgePass: credentials.user.userId,
            bridgeUrl: config_service_1.ConfigService.instance.get('NETWORK_URL'),
            encryptionKey: credentials.user.mnemonic,
            appDetails: sdk_manager_service_1.SdkManager.getAppDetails(),
        });
        const networkFacade = new network_facade_service_1.NetworkFacade(networkModule, environment, download_service_1.DownloadService.instance, crypto_service_1.CryptoService.instance);
        return networkFacade;
    };
    registerStartMiddlewares = () => {
        this.app.use((0, auth_middleware_1.AuthMiddleware)(auth_service_1.AuthService.instance));
        this.app.use((0, request_logger_middleware_1.RequestLoggerMiddleware)({
            enable: true,
        }));
        this.app.use(body_parser_1.default.text({ type: ['application/xml', 'text/xml'] }));
        this.app.use(mkcol_middleware_1.MkcolMiddleware);
    };
    registerEndMiddleWares = () => {
        this.app.use(errors_middleware_1.ErrorHandlingMiddleware);
    };
    registerHandlers = async () => {
        const serverListenPath = /(.*)/;
        const networkFacade = await this.getNetworkFacade();
        const webDavFolderService = new webdav_folder_service_1.WebDavFolderService({
            driveFolderService: this.driveFolderService,
            configService: this.configService,
        });
        this.app.head(serverListenPath, (0, express_async_handler_1.default)(new HEAD_handler_1.HEADRequestHandler({
            driveFileService: this.driveFileService,
        }).handle));
        this.app.get(serverListenPath, (0, express_async_handler_1.default)(new GET_handler_1.GETRequestHandler({
            driveFileService: this.driveFileService,
            downloadService: this.downloadService,
            cryptoService: this.cryptoService,
            authService: this.authService,
            networkFacade: networkFacade,
        }).handle));
        this.app.options(serverListenPath, (0, express_async_handler_1.default)(new OPTIONS_handler_1.OPTIONSRequestHandler().handle));
        this.app.propfind(serverListenPath, (0, express_async_handler_1.default)(new PROPFIND_handler_1.PROPFINDRequestHandler({
            driveFileService: this.driveFileService,
            driveFolderService: this.driveFolderService,
        }).handle));
        this.app.put(serverListenPath, (0, express_async_handler_1.default)(new PUT_handler_1.PUTRequestHandler({
            driveFileService: this.driveFileService,
            driveFolderService: this.driveFolderService,
            webDavFolderService: webDavFolderService,
            authService: this.authService,
            trashService: this.trashService,
            networkFacade: networkFacade,
        }).handle));
        this.app.mkcol(serverListenPath, (0, express_async_handler_1.default)(new MKCOL_handler_1.MKCOLRequestHandler({
            driveFolderService: this.driveFolderService,
            webDavFolderService: webDavFolderService,
        }).handle));
        this.app.delete(serverListenPath, (0, express_async_handler_1.default)(new DELETE_handler_1.DELETERequestHandler({
            trashService: this.trashService,
            driveFileService: this.driveFileService,
            driveFolderService: this.driveFolderService,
        }).handle));
        this.app.proppatch(serverListenPath, (0, express_async_handler_1.default)(new PROPPATCH_handler_1.PROPPATCHRequestHandler().handle));
        this.app.move(serverListenPath, (0, express_async_handler_1.default)(new MOVE_handler_1.MOVERequestHandler({
            driveFolderService: this.driveFolderService,
            driveFileService: this.driveFileService,
            webDavFolderService,
        }).handle));
        this.app.copy(serverListenPath, (0, express_async_handler_1.default)(new COPY_handler_1.COPYRequestHandler().handle));
    };
    start = async () => {
        const configs = await this.configService.readWebdavConfig();
        this.app.disable('x-powered-by');
        this.registerStartMiddlewares();
        await this.registerHandlers();
        this.registerEndMiddleWares();
        const plainHttp = configs.protocol === 'http';
        let server;
        if (plainHttp) {
            server = http_1.default.createServer(this.app);
        }
        else {
            const httpsCerts = await network_utils_1.NetworkUtils.getWebdavSSLCerts(configs);
            server = https_1.default.createServer(httpsCerts, this.app);
        }
        server.requestTimeout = configs.timeoutMinutes * 60 * 1000;
        server.listen(Number(configs.port), configs.host, undefined, () => {
            logger_utils_1.webdavLogger.info(`Internxt ${sdk_manager_service_1.SdkManager.getAppDetails().clientVersion} WebDav server ` +
                `listening at ${configs.protocol}://${configs.host}:${configs.port}`);
        });
    };
}
exports.WebDavServer = WebDavServer;
