"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PUTRequestHandler = void 0;
const drive_file_service_1 = require("../../services/drive/drive-file.service");
const errors_utils_1 = require("../../utils/errors.utils");
const webdav_utils_1 = require("../../utils/webdav.utils");
const logger_utils_1 = require("../../utils/logger.utils");
const types_1 = require("@internxt/sdk/dist/drive/storage/types");
const cli_utils_1 = require("../../utils/cli.utils");
const stream_utils_1 = require("../../utils/stream.utils");
const thumbnail_utils_1 = require("../../utils/thumbnail.utils");
const thumbnail_service_1 = require("../../services/thumbnail.service");
const async_utils_1 = require("../../utils/async.utils");
class PUTRequestHandler {
    dependencies;
    constructor(dependencies) {
        this.dependencies = dependencies;
    }
    handle = async (req, res) => {
        const contentLength = Number(req.headers['content-length']);
        if (!contentLength || isNaN(contentLength) || contentLength <= 0) {
            throw new errors_utils_1.UnsupportedMediaTypeError('Empty files are not supported');
        }
        const resource = await webdav_utils_1.WebDavUtils.getRequestedResource(req.url);
        const driveFileItem = await webdav_utils_1.WebDavUtils.getDriveItemFromResource({
            resource: resource,
            driveFileService: this.dependencies.driveFileService,
            driveFolderService: this.dependencies.driveFolderService,
        });
        if (driveFileItem?.itemType === 'folder') {
            throw new errors_utils_1.NotFoundError('Folders cannot be created with PUT. Use MKCOL instead.');
        }
        logger_utils_1.webdavLogger.info(`[PUT] Request received for file at ${resource.url}`);
        logger_utils_1.webdavLogger.info(`[PUT] Uploading '${resource.name}' to '${resource.parentPath}'`);
        const parentDriveFolderItem = (await this.dependencies.webDavFolderService.getDriveFolderItemFromPath(resource.parentPath)) ??
            (await this.dependencies.webDavFolderService.createParentPathOrThrow(resource.parentPath));
        try {
            if (driveFileItem && driveFileItem.status === 'EXISTS') {
                logger_utils_1.webdavLogger.info(`[PUT] File '${resource.name}' already exists in '${resource.path.dir}', trashing it...`);
                await this.dependencies.trashService.trashItems({
                    items: [{ type: driveFileItem.itemType, uuid: driveFileItem.uuid }],
                });
            }
        }
        catch {
        }
        const { user } = await this.dependencies.authService.getAuthDetails();
        const fileType = resource.path.ext.replace('.', '');
        const timer = cli_utils_1.CLIUtils.timer();
        let bufferStream;
        let fileStream = req;
        const isThumbnailable = (0, thumbnail_utils_1.isFileThumbnailable)(fileType);
        if (isThumbnailable) {
            bufferStream = new stream_utils_1.BufferStream();
            fileStream = req.pipe(bufferStream);
        }
        let uploaded = false, aborted = false;
        const progressCallback = (progress) => {
            if (!uploaded && !aborted) {
                logger_utils_1.webdavLogger.info(`[PUT] Upload progress for file ${resource.name}: ${(progress * 100).toFixed(2)}%`);
            }
        };
        const fileId = await new Promise((resolve, reject) => {
            const state = this.dependencies.networkFacade.uploadFile(fileStream, contentLength, user.bucket, (err, res) => {
                if (err) {
                    aborted = true;
                    return reject(err);
                }
                resolve(res);
            }, progressCallback);
            res.on('close', async () => {
                aborted = true;
                if (!uploaded) {
                    logger_utils_1.webdavLogger.info('[PUT] ❌ HTTP Client has been disconnected, res has been closed.');
                    state.stop();
                }
            });
        });
        uploaded = true;
        logger_utils_1.webdavLogger.info('[PUT] ✅ File uploaded to network');
        const file = await drive_file_service_1.DriveFileService.instance.createFile({
            plainName: resource.path.name,
            type: fileType,
            size: contentLength,
            folderUuid: parentDriveFolderItem.uuid,
            fileId: fileId,
            bucket: user.bucket,
            encryptVersion: types_1.EncryptionVersion.Aes03,
        });
        try {
            if (isThumbnailable && bufferStream) {
                const thumbnailBuffer = bufferStream.getBuffer();
                if (thumbnailBuffer) {
                    await thumbnail_service_1.ThumbnailService.instance.uploadThumbnail(thumbnailBuffer, fileType, user.bucket, file.uuid, this.dependencies.networkFacade);
                }
            }
        }
        catch (error) {
            logger_utils_1.webdavLogger.info(`[PUT] ❌ File thumbnail upload failed ${error.message}`);
        }
        const uploadTime = timer.stop();
        logger_utils_1.webdavLogger.info(`[PUT] ✅ File uploaded in ${uploadTime}ms to Internxt Drive`);
        await async_utils_1.AsyncUtils.sleep(500);
        logger_utils_1.webdavLogger.info(`[PUT] [RESPONSE-201] ${resource.url} - Returning 201 Created after ${uploadTime}ms (+ 500ms propagation delay)`);
        res.status(201).send();
    };
}
exports.PUTRequestHandler = PUTRequestHandler;
