"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BufferStream = exports.StreamUtils = void 0;
const node_stream_1 = require("node:stream");
class StreamUtils {
    static readStreamToReadableStream(readStream) {
        const readable = new ReadableStream({
            async start(controller) {
                readStream.on('data', (chunk) => {
                    controller.enqueue(Buffer.from(chunk));
                });
                readStream.on('end', () => {
                    controller.close();
                });
            },
        });
        return readable;
    }
    static writeStreamToWritableStream(writeStream) {
        const writable = new WritableStream({
            write(chunk) {
                writeStream.write(chunk);
            },
        });
        return writable;
    }
    static joinReadableBinaryStreams(streams) {
        const streamsCopy = streams.map((s) => s);
        let keepReading = true;
        const flush = () => streamsCopy.forEach((s) => s.cancel());
        const stream = new ReadableStream({
            async pull(controller) {
                if (!keepReading)
                    return flush();
                const downStream = streamsCopy.shift();
                if (!downStream) {
                    return controller.close();
                }
                const reader = downStream.getReader();
                let done = false;
                while (!done && keepReading) {
                    const status = await reader.read();
                    if (!status.done) {
                        controller.enqueue(status.value);
                    }
                    done = status.done;
                }
                reader.releaseLock();
            },
            cancel() {
                keepReading = false;
            },
        });
        return stream;
    }
}
exports.StreamUtils = StreamUtils;
class BufferStream extends node_stream_1.Transform {
    buffer;
    constructor(opts) {
        super(opts);
        this.buffer = null;
    }
    _transform(chunk, _, callback) {
        const currentBuffer = this.buffer ?? Buffer.alloc(0);
        this.buffer = Buffer.concat([currentBuffer, chunk]);
        callback(null, chunk);
    }
    _flush(callback) {
        callback();
    }
    reset() {
        this.buffer = null;
    }
    getBuffer() {
        return this.buffer;
    }
}
exports.BufferStream = BufferStream;
