"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkUtils = void 0;
const node_crypto_1 = require("node:crypto");
const promises_1 = require("node:fs/promises");
const node_path_1 = __importDefault(require("node:path"));
const selfsigned_1 = __importDefault(require("selfsigned"));
const range_parser_1 = __importDefault(require("range-parser"));
const config_service_1 = require("../services/config.service");
class NetworkUtils {
    static getAuthFromCredentials(creds) {
        return {
            username: creds.user,
            password: (0, node_crypto_1.createHash)('SHA256').update(Buffer.from(creds.pass)).digest().toString('hex'),
        };
    }
    static WEBDAV_SSL_CERTS_PATH = {
        cert: node_path_1.default.join(config_service_1.ConfigService.WEBDAV_SSL_CERTS_DIR, 'cert.crt'),
        privateKey: node_path_1.default.join(config_service_1.ConfigService.WEBDAV_SSL_CERTS_DIR, 'priv.key'),
    };
    static async generateNewSelfsignedCerts(configs) {
        const newCerts = await this.generateSelfSignedSSLCerts(configs);
        this.saveWebdavSSLCerts(newCerts);
        return {
            cert: newCerts.cert,
            key: newCerts.private,
        };
    }
    static async getWebdavSSLCerts(configs) {
        const existCert = !!(await (0, promises_1.stat)(this.WEBDAV_SSL_CERTS_PATH.cert).catch(() => false));
        const existKey = !!(await (0, promises_1.stat)(this.WEBDAV_SSL_CERTS_PATH.privateKey).catch(() => false));
        if (!existCert || !existKey) {
            return this.generateNewSelfsignedCerts(configs);
        }
        else {
            let selfsignedCert = {
                cert: await (0, promises_1.readFile)(this.WEBDAV_SSL_CERTS_PATH.cert),
                key: await (0, promises_1.readFile)(this.WEBDAV_SSL_CERTS_PATH.privateKey),
            };
            const { validTo } = new node_crypto_1.X509Certificate(selfsignedCert.cert);
            const dateToday = new Date();
            const dateValid = new Date(validTo);
            if (dateToday > dateValid) {
                const newCerts = await this.generateNewSelfsignedCerts(configs);
                selfsignedCert = {
                    cert: newCerts.cert,
                    key: newCerts.key,
                };
            }
            return selfsignedCert;
        }
    }
    static async saveWebdavSSLCerts(pems) {
        await (0, promises_1.writeFile)(this.WEBDAV_SSL_CERTS_PATH.cert, pems.cert, 'utf8');
        await (0, promises_1.writeFile)(this.WEBDAV_SSL_CERTS_PATH.privateKey, pems.private, 'utf8');
    }
    static async generateSelfSignedSSLCerts(configs) {
        const attrs = [{ name: 'commonName', value: configs.host }];
        const extensions = [
            {
                name: 'subjectAltName',
                altNames: [
                    {
                        type: 2,
                        value: configs.host,
                    },
                ],
            },
        ];
        const notAfterDate = new Date();
        notAfterDate.setDate(notAfterDate.getDate() + 365);
        const pems = await selfsigned_1.default.generate(attrs, { notAfterDate, algorithm: 'sha256', keySize: 2048, extensions });
        return pems;
    }
    static parseRangeHeader(rangeOptions) {
        if (!rangeOptions.range) {
            return;
        }
        const parsed = (0, range_parser_1.default)(rangeOptions.totalFileSize, rangeOptions.range);
        if (Array.isArray(parsed)) {
            if (parsed.length > 1) {
                throw new Error(`Multi Range-Requests functionality is not implemented. ${JSON.stringify(rangeOptions)}`);
            }
            else if (parsed.length <= 0) {
                throw new Error(`Empty Range-Request. ${JSON.stringify(rangeOptions)}`);
            }
            else if (parsed.type !== 'bytes') {
                throw new Error(`Unkwnown Range-Request type "${parsed.type}". ${JSON.stringify(rangeOptions)}`);
            }
            else {
                const rangeSize = parsed[0].end - parsed[0].start + 1;
                return {
                    range: rangeOptions.range,
                    rangeSize: rangeSize,
                    totalFileSize: rangeOptions.totalFileSize,
                    parsed: parsed[0],
                };
            }
        }
        else if (parsed === -1) {
            throw new Error(`Malformed Range-Request. ${JSON.stringify(rangeOptions)}`);
        }
        else if (parsed === -2) {
            throw new Error(`Unsatisfiable Range-Request. ${JSON.stringify(rangeOptions)}`);
        }
        else {
            throw new Error(`Unknown error from Range-Request. ${JSON.stringify(rangeOptions)}`);
        }
    }
}
exports.NetworkUtils = NetworkUtils;
