"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NoFlagProvidedError = exports.CLIUtils = void 0;
const core_1 = require("@oclif/core");
const cli_progress_1 = __importDefault(require("cli-progress"));
const tty_table_1 = __importDefault(require("tty-table"));
const command_types_1 = require("../types/command.types");
const inquirer_utils_1 = require("./inquirer.utils");
const errors_utils_1 = require("./errors.utils");
const validation_service_1 = require("../services/validation.service");
const sdk_manager_service_1 = require("../services/sdk-manager.service");
const inxt_js_1 = require("@internxt/inxt-js");
const config_service_1 = require("../services/config.service");
const network_facade_service_1 = require("../services/network/network-facade.service");
const download_service_1 = require("../services/network/download.service");
const crypto_service_1 = require("../services/crypto.service");
class CLIUtils {
    static clearPreviousLine = (jsonFlag) => {
        if (!jsonFlag) {
            process.stdout?.write?.('\x1b[1A');
            process.stdout?.clearLine?.(0);
        }
    };
    static warning = (reporter, message) => {
        reporter(core_1.ux.colorize('#a67805', `⚠ Warning: ${message}`));
    };
    static error = (reporter, message) => {
        reporter(core_1.ux.colorize('red', `⚠ Error: ${message}`));
    };
    static success = (reporter, message) => {
        reporter(core_1.ux.colorize('green', `✓ ${message}`));
    };
    static log = (reporter, message) => {
        reporter(`${message}`);
    };
    static consoleLog = (message) => {
        console.log(message);
    };
    static doing = (message, jsonFlag) => {
        if (!jsonFlag) {
            core_1.ux.action.start(message, undefined, {});
        }
    };
    static done = (jsonFlag) => {
        if (!jsonFlag && core_1.ux.action.running) {
            core_1.ux.action.stop(core_1.ux.colorize('green', 'done ✓'));
        }
    };
    static failed = (jsonFlag) => {
        if (!jsonFlag && core_1.ux.action.running) {
            core_1.ux.action.stop(core_1.ux.colorize('red', 'failed ✕'));
        }
    };
    static progress = (opts, jsonFlag) => {
        if (!jsonFlag) {
            return new cli_progress_1.default.SingleBar({ noTTYOutput: Boolean(!process.stdin.isTTY), ...opts }, cli_progress_1.default.Presets.shades_classic);
        }
    };
    static table = (reporter, header, rows) => {
        const table = (0, tty_table_1.default)(header, rows);
        reporter(table.render());
    };
    static generateTableHeaderFromType = () => {
        return Object.keys({}).map((key) => ({
            value: key,
            alias: key.charAt(0).toUpperCase() + key.slice(1),
        }));
    };
    static CommonFlags = {
        'non-interactive': core_1.Flags.boolean({
            char: 'x',
            env: 'INXT_NONINTERACTIVE',
            helpGroup: 'helper',
            description: 'Prevents the CLI from being interactive. When enabled, the CLI will not request input through the console and will throw errors directly.',
            required: false,
        }),
    };
    static getValueFromFlag = async (flag, command, validation, reporter) => {
        if (flag.value) {
            if (validation.canBeEmpty && flag.value.trim().length === 0) {
                return '';
            }
            const isValid = await validation.validate(flag.value);
            if (isValid) {
                return flag.value;
            }
            else if (command.nonInteractive) {
                throw validation.error;
            }
            else {
                CLIUtils.error(reporter, validation.error.message);
            }
        }
        if (command.nonInteractive) {
            throw new NoFlagProvidedError(flag.name);
        }
        else {
            const maxAttempts = command.maxAttempts ?? 3;
            return await CLIUtils.promptWithAttempts(command.prompt, maxAttempts, validation, reporter);
        }
    };
    static getDestinationFolderUuid = async ({ destinationFolderUuidFlag, destinationFlagName, nonInteractive, reporter, }) => {
        const destinationFolderUuid = await this.getValueFromFlag({
            value: destinationFolderUuidFlag,
            name: destinationFlagName,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the destination folder id? (leave empty for the root folder)',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFolderUuidError(),
            canBeEmpty: true,
        }, reporter);
        if (destinationFolderUuid.trim().length === 0) {
            return undefined;
        }
        else {
            return destinationFolderUuid;
        }
    };
    static promptWithAttempts = async (prompt, maxAttempts, validation, reporter) => {
        let isValid = false;
        let currentAttempts = 0;
        let promptValue = '';
        do {
            promptValue = await inquirer_utils_1.InquirerUtils.prompt(prompt.message, prompt.options);
            if (validation.canBeEmpty) {
                if (!promptValue || promptValue.trim().length === 0) {
                    return '';
                }
            }
            isValid = await validation.validate(promptValue);
            if (!isValid) {
                currentAttempts++;
                if (currentAttempts < maxAttempts) {
                    CLIUtils.warning(reporter, validation.error.message + ', please type it again');
                }
            }
        } while (!isValid && currentAttempts < maxAttempts);
        if (!isValid) {
            throw validation.error;
        }
        else {
            return promptValue;
        }
    };
    static timer = () => {
        const start = new Date();
        return {
            stop: () => {
                const end = new Date();
                const time = end.getTime() - start.getTime();
                return time;
            },
        };
    };
    static catchError = ({ error, logReporter, command, jsonFlag, }) => {
        let message;
        if ('message' in error && error.message.trim().length > 0) {
            message = error.message;
        }
        else {
            message = JSON.stringify(error);
        }
        CLIUtils.failed(jsonFlag);
        if (jsonFlag) {
            CLIUtils.consoleLog(JSON.stringify({ success: false, message }));
        }
        else {
            errors_utils_1.ErrorUtils.report(error, { command });
            CLIUtils.error(logReporter, message);
        }
    };
    static parseEmpty = async (input) => (input.trim().length === 0 ? ' ' : input);
    static prepareNetwork = ({ jsonFlag, loginUserDetails, }) => {
        CLIUtils.doing('Preparing Network', jsonFlag);
        const networkModule = sdk_manager_service_1.SdkManager.instance.getNetwork({
            user: loginUserDetails.bridgeUser,
            pass: loginUserDetails.userId,
        });
        const environment = new inxt_js_1.Environment({
            bridgeUser: loginUserDetails.bridgeUser,
            bridgePass: loginUserDetails.userId,
            bridgeUrl: config_service_1.ConfigService.instance.get('NETWORK_URL'),
            encryptionKey: loginUserDetails.mnemonic,
            appDetails: sdk_manager_service_1.SdkManager.getAppDetails(),
        });
        const networkFacade = new network_facade_service_1.NetworkFacade(networkModule, environment, download_service_1.DownloadService.instance, crypto_service_1.CryptoService.instance);
        CLIUtils.done(jsonFlag);
        return networkFacade;
    };
}
exports.CLIUtils = CLIUtils;
class NoFlagProvidedError extends Error {
    constructor(flag) {
        super(`No ${flag} flag has been provided`);
        Object.setPrototypeOf(this, NoFlagProvidedError.prototype);
    }
}
exports.NoFlagProvidedError = NoFlagProvidedError;
