"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationService = void 0;
const lib_1 = require("@internxt/lib");
const bip39_1 = require("bip39");
const promises_1 = __importDefault(require("node:fs/promises"));
class ValidationService {
    static instance = new ValidationService();
    validateEmail = (email) => {
        return lib_1.auth.isValidEmail(email);
    };
    validate2FA = (code) => {
        return /^\d{6}$/.test(code);
    };
    validateMnemonic = (mnemonic) => {
        return (0, bip39_1.validateMnemonic)(mnemonic);
    };
    validateUUIDv4 = (uuid) => {
        return /^[0-9A-F]{8}-[0-9A-F]{4}-4[0-9A-F]{3}-[89AB][0-9A-F]{3}-[0-9A-F]{12}$/i.test(uuid);
    };
    validateYesOrNoString = (message) => {
        return message.length > 0 && /^(yes|no|y|n)$/i.test(message.trim());
    };
    validateTCPIntegerPort = (port) => {
        return /^\d+$/.test(port) && Number(port) >= 1 && Number(port) <= 65535;
    };
    validateStringIsNotEmpty = (str) => {
        return str.trim().length > 0;
    };
    validateDirectoryExists = async (path) => {
        const directoryStat = await promises_1.default.stat(path);
        return directoryStat.isDirectory();
    };
    validateFileExists = async (path) => {
        const fileStat = await promises_1.default.stat(path);
        return fileStat.isFile();
    };
    validateTokenAndCheckExpiration = (token) => {
        if (!token || typeof token !== 'string') {
            return { isValid: false, expiration: { expired: true, refreshRequired: false } };
        }
        const parts = token.split('.');
        if (parts.length !== 3) {
            return { isValid: false, expiration: { expired: true, refreshRequired: false } };
        }
        try {
            const payload = JSON.parse(atob(parts[1]));
            if (typeof payload.exp !== 'number') {
                return { isValid: false, expiration: { expired: true, refreshRequired: false } };
            }
            const currentTime = Math.floor(Date.now() / 1000);
            const twoDaysInSeconds = 2 * 24 * 60 * 60;
            const remainingSeconds = payload.exp - currentTime;
            const expired = remainingSeconds <= 0;
            const refreshRequired = remainingSeconds > 0 && remainingSeconds <= twoDaysInSeconds;
            return { isValid: true, expiration: { expired, refreshRequired } };
        }
        catch {
            return { isValid: false, expiration: { expired: true, refreshRequired: false } };
        }
    };
}
exports.ValidationService = ValidationService;
