"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UploadFolderService = void 0;
const node_path_1 = require("node:path");
const errors_utils_1 = require("../../../utils/errors.utils");
const logger_utils_1 = require("../../../utils/logger.utils");
const drive_folder_service_1 = require("../../drive/drive-folder.service");
const upload_types_1 = require("./upload.types");
class UploadFolderService {
    static instance = new UploadFolderService();
    async createFolders({ foldersToCreate, destinationFolderUuid, currentProgress, emitProgress, }) {
        const folderMap = new Map();
        for (const folder of foldersToCreate) {
            const parentPath = (0, node_path_1.dirname)(folder.relativePath);
            const parentUuid = parentPath === '.' ? destinationFolderUuid : folderMap.get(parentPath);
            if (!parentUuid) {
                logger_utils_1.logger.warn(`Parent folder not found for ${folder.relativePath}, skipping...`);
                continue;
            }
            const createdFolderUuid = await this.createFolderWithRetry({
                folderName: folder.name,
                parentFolderUuid: parentUuid,
            });
            if (createdFolderUuid) {
                folderMap.set(folder.relativePath, createdFolderUuid);
                currentProgress.itemsUploaded++;
                emitProgress();
            }
        }
        return folderMap;
    }
    async createFolderWithRetry({ folderName, parentFolderUuid }) {
        for (let attempt = 0; attempt <= upload_types_1.MAX_RETRIES; attempt++) {
            try {
                const [createFolderPromise] = drive_folder_service_1.DriveFolderService.instance.createFolder({
                    plainName: folderName,
                    parentFolderUuid,
                });
                const createdFolder = await createFolderPromise;
                return createdFolder.uuid;
            }
            catch (error) {
                if ((0, errors_utils_1.isAlreadyExistsError)(error)) {
                    logger_utils_1.logger.info(`Folder ${folderName} already exists, skipping...`);
                    return null;
                }
                if (attempt < upload_types_1.MAX_RETRIES) {
                    const delay = upload_types_1.DELAYS_MS[attempt];
                    logger_utils_1.logger.warn(`Failed to create folder ${folderName},
                retrying in ${delay}ms... (attempt ${attempt + 1}/${upload_types_1.MAX_RETRIES + 1})`);
                    await new Promise((resolve) => setTimeout(resolve, delay));
                }
                else {
                    logger_utils_1.logger.error(`Failed to create folder ${folderName} after ${upload_types_1.MAX_RETRIES + 1} attempts`);
                    throw error;
                }
            }
        }
        return null;
    }
}
exports.UploadFolderService = UploadFolderService;
