"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UploadFileService = void 0;
const logger_utils_1 = require("../../../utils/logger.utils");
const upload_types_1 = require("./upload.types");
const drive_file_service_1 = require("../../drive/drive-file.service");
const node_path_1 = require("node:path");
const errors_utils_1 = require("../../../utils/errors.utils");
const promises_1 = require("node:fs/promises");
const types_1 = require("@internxt/sdk/dist/drive/storage/types");
const thumbnail_utils_1 = require("../../../utils/thumbnail.utils");
class UploadFileService {
    static instance = new UploadFileService();
    async uploadFilesInChunks({ network, filesToUpload, folderMap, bucket, destinationFolderUuid, currentProgress, emitProgress, }) {
        let bytesUploaded = 0;
        const chunks = this.chunkArray(filesToUpload, upload_types_1.MAX_CONCURRENT_UPLOADS);
        for (const chunk of chunks) {
            await Promise.allSettled(chunk.map(async (file) => {
                const parentPath = (0, node_path_1.dirname)(file.relativePath);
                const parentFolderUuid = parentPath === '.' || parentPath === '' ? destinationFolderUuid : folderMap.get(parentPath);
                if (!parentFolderUuid) {
                    logger_utils_1.logger.warn(`Parent folder not found for ${file.relativePath}, skipping...`);
                    return null;
                }
                const createdFileUuid = await this.uploadFileWithRetry({
                    file,
                    network,
                    bucket,
                    parentFolderUuid,
                });
                if (createdFileUuid) {
                    bytesUploaded += file.size;
                    currentProgress.bytesUploaded += file.size;
                    currentProgress.itemsUploaded++;
                }
                emitProgress();
            }));
        }
        return bytesUploaded;
    }
    async uploadFileWithRetry({ file, network, bucket, parentFolderUuid, }) {
        for (let attempt = 0; attempt <= upload_types_1.MAX_RETRIES; attempt++) {
            try {
                const stats = await (0, promises_1.stat)(file.absolutePath);
                if (!stats.size) {
                    logger_utils_1.logger.warn(`Skipping empty file: ${file.relativePath}`);
                    return null;
                }
                const fileType = (0, node_path_1.extname)(file.absolutePath).replaceAll('.', '');
                const { fileStream, bufferStream } = (0, thumbnail_utils_1.createFileStreamWithBuffer)({
                    path: file.absolutePath,
                    fileType,
                });
                const fileId = await new Promise((resolve, reject) => {
                    network.uploadFile(fileStream, stats.size, bucket, (err, res) => {
                        if (err) {
                            return reject(err);
                        }
                        resolve(res);
                    }, () => { });
                });
                const createdDriveFile = await drive_file_service_1.DriveFileService.instance.createFile({
                    plainName: file.name,
                    type: fileType,
                    size: stats.size,
                    folderUuid: parentFolderUuid,
                    fileId,
                    bucket,
                    encryptVersion: types_1.EncryptionVersion.Aes03,
                    creationTime: stats.birthtime?.toISOString(),
                    modificationTime: stats.mtime?.toISOString(),
                });
                if (bufferStream) {
                    void (0, thumbnail_utils_1.tryUploadThumbnail)({
                        bufferStream,
                        fileType,
                        userBucket: bucket,
                        fileUuid: createdDriveFile.uuid,
                        networkFacade: network,
                    });
                }
                return createdDriveFile.fileId;
            }
            catch (error) {
                if ((0, errors_utils_1.isAlreadyExistsError)(error)) {
                    const msg = `File ${file.name} already exists, skipping...`;
                    logger_utils_1.logger.info(msg);
                    return null;
                }
                if (attempt < upload_types_1.MAX_RETRIES) {
                    const delay = upload_types_1.DELAYS_MS[attempt];
                    const retryMsg = `Failed to upload file ${file.name}, retrying in ${delay}ms...`;
                    logger_utils_1.logger.warn(`${retryMsg} (attempt ${attempt + 1}/${upload_types_1.MAX_RETRIES + 1})`);
                    await new Promise((resolve) => setTimeout(resolve, delay));
                }
                else {
                    logger_utils_1.logger.error(`Failed to upload file ${file.name} after ${upload_types_1.MAX_RETRIES + 1} attempts`);
                    return null;
                }
            }
        }
        return null;
    }
    chunkArray(array, chunkSize) {
        const chunks = [];
        for (let i = 0; i < array.length; i += chunkSize) {
            chunks.push(array.slice(i, i + chunkSize));
        }
        return chunks;
    }
}
exports.UploadFileService = UploadFileService;
