"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LocalFilesystemService = void 0;
const node_fs_1 = require("node:fs");
const node_path_1 = require("node:path");
const logger_utils_1 = require("../../utils/logger.utils");
class LocalFilesystemService {
    static instance = new LocalFilesystemService();
    async scanLocalDirectory(path) {
        const folders = [];
        const files = [];
        const parentPath = (0, node_path_1.dirname)(path);
        const totalBytes = await this.scanRecursive(path, parentPath, folders, files);
        return {
            folders,
            files,
            totalItems: folders.length + files.length,
            totalBytes,
        };
    }
    async scanRecursive(currentPath, parentPath, folders, files) {
        try {
            const stats = await node_fs_1.promises.stat(currentPath);
            const relativePath = (0, node_path_1.relative)(parentPath, currentPath);
            if (stats.isFile() && stats.size > 0) {
                const fileInfo = (0, node_path_1.parse)(currentPath);
                files.push({
                    type: 'file',
                    name: fileInfo.name,
                    absolutePath: currentPath,
                    relativePath,
                    size: stats.size,
                });
                return stats.size;
            }
            if (stats.isDirectory()) {
                folders.push({
                    type: 'folder',
                    name: (0, node_path_1.basename)(currentPath),
                    absolutePath: currentPath,
                    relativePath,
                    size: 0,
                });
                const entries = await node_fs_1.promises.readdir(currentPath, { withFileTypes: true });
                const validEntries = entries.filter((e) => !e.isSymbolicLink());
                const bytesArray = await Promise.all(validEntries.map((e) => this.scanRecursive((0, node_path_1.join)(currentPath, e.name), parentPath, folders, files)));
                return bytesArray.reduce((sum, bytes) => sum + bytes, 0);
            }
            return 0;
        }
        catch (error) {
            logger_utils_1.logger.warn(`Error scanning path ${currentPath}: ${error.message} - skipping...`);
            return 0;
        }
    }
}
exports.LocalFilesystemService = LocalFilesystemService;
