"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigService = void 0;
const node_path_1 = __importDefault(require("node:path"));
const os_1 = __importDefault(require("os"));
const promises_1 = __importDefault(require("node:fs/promises"));
const crypto_service_1 = require("./crypto.service");
class ConfigService {
    static INTERNXT_CLI_DATA_DIR = node_path_1.default.join(os_1.default.homedir(), '.internxt-cli');
    static INTERNXT_CLI_LOGS_DIR = node_path_1.default.join(this.INTERNXT_CLI_DATA_DIR, 'logs');
    static INTERNXT_TMP_DIR = os_1.default.tmpdir();
    static CREDENTIALS_FILE = node_path_1.default.join(this.INTERNXT_CLI_DATA_DIR, '.inxtcli');
    static DRIVE_SQLITE_FILE = node_path_1.default.join(this.INTERNXT_CLI_DATA_DIR, 'internxt-cli-drive.sqlite');
    static WEBDAV_SSL_CERTS_DIR = node_path_1.default.join(this.INTERNXT_CLI_DATA_DIR, 'certs');
    static WEBDAV_CONFIGS_FILE = node_path_1.default.join(this.INTERNXT_CLI_DATA_DIR, 'config.webdav.inxt');
    static WEBDAV_DEFAULT_HOST = '127.0.0.1';
    static WEBDAV_DEFAULT_PORT = '3005';
    static WEBDAV_DEFAULT_PROTOCOL = 'https';
    static WEBDAV_DEFAULT_TIMEOUT = 0;
    static WEBDAV_DEFAULT_CREATE_FULL_PATH = true;
    static instance = new ConfigService();
    get = (key) => {
        const value = process.env[key];
        if (!value)
            throw new Error(`Config key ${key} was not found in process.env`);
        return value;
    };
    saveUser = async (loginCredentials) => {
        await this.ensureInternxtCliDataDirExists();
        const credentialsString = JSON.stringify(loginCredentials);
        const encryptedCredentials = crypto_service_1.CryptoService.instance.encryptText(credentialsString);
        await promises_1.default.writeFile(ConfigService.CREDENTIALS_FILE, encryptedCredentials, 'utf8');
    };
    clearUser = async () => {
        const stat = await promises_1.default.stat(ConfigService.CREDENTIALS_FILE);
        if (stat.size === 0)
            throw new Error('Credentials file is already empty');
        return promises_1.default.writeFile(ConfigService.CREDENTIALS_FILE, '', 'utf8');
    };
    readUser = async () => {
        try {
            const encryptedCredentials = await promises_1.default.readFile(ConfigService.CREDENTIALS_FILE, 'utf8');
            const credentialsString = crypto_service_1.CryptoService.instance.decryptText(encryptedCredentials);
            const loginCredentials = JSON.parse(credentialsString);
            return loginCredentials;
        }
        catch {
            return;
        }
    };
    saveWebdavConfig = async (webdavConfig) => {
        await this.ensureInternxtCliDataDirExists();
        const configs = JSON.stringify(webdavConfig);
        await promises_1.default.writeFile(ConfigService.WEBDAV_CONFIGS_FILE, configs, 'utf8');
    };
    readWebdavConfig = async () => {
        try {
            const configsData = await promises_1.default.readFile(ConfigService.WEBDAV_CONFIGS_FILE, 'utf8');
            const configs = JSON.parse(configsData);
            return {
                host: configs?.host ?? ConfigService.WEBDAV_DEFAULT_HOST,
                port: configs?.port ?? ConfigService.WEBDAV_DEFAULT_PORT,
                protocol: configs?.protocol ?? ConfigService.WEBDAV_DEFAULT_PROTOCOL,
                timeoutMinutes: configs?.timeoutMinutes ?? ConfigService.WEBDAV_DEFAULT_TIMEOUT,
                createFullPath: configs?.createFullPath ?? ConfigService.WEBDAV_DEFAULT_CREATE_FULL_PATH,
            };
        }
        catch {
            return {
                host: ConfigService.WEBDAV_DEFAULT_HOST,
                port: ConfigService.WEBDAV_DEFAULT_PORT,
                protocol: ConfigService.WEBDAV_DEFAULT_PROTOCOL,
                timeoutMinutes: ConfigService.WEBDAV_DEFAULT_TIMEOUT,
                createFullPath: ConfigService.WEBDAV_DEFAULT_CREATE_FULL_PATH,
            };
        }
    };
    ensureInternxtCliDataDirExists = async () => {
        try {
            await promises_1.default.access(ConfigService.INTERNXT_CLI_DATA_DIR);
        }
        catch {
            await promises_1.default.mkdir(ConfigService.INTERNXT_CLI_DATA_DIR);
        }
    };
    ensureWebdavCertsDirExists = async () => {
        try {
            await promises_1.default.access(ConfigService.WEBDAV_SSL_CERTS_DIR);
        }
        catch {
            await promises_1.default.mkdir(ConfigService.WEBDAV_SSL_CERTS_DIR);
        }
    };
    ensureInternxtLogsDirExists = async () => {
        try {
            await promises_1.default.access(ConfigService.INTERNXT_CLI_LOGS_DIR);
        }
        catch {
            await promises_1.default.mkdir(ConfigService.INTERNXT_CLI_LOGS_DIR);
        }
    };
}
exports.ConfigService = ConfigService;
