"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthService = void 0;
const sdk_manager_service_1 = require("./sdk-manager.service");
const crypto_service_1 = require("./crypto.service");
const config_service_1 = require("./config.service");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("./validation.service");
class AuthService {
    static instance = new AuthService();
    doLogin = async (email, password, twoFactorCode) => {
        const authClient = sdk_manager_service_1.SdkManager.instance.getAuth();
        const loginDetails = {
            email: email.toLowerCase(),
            password: password,
            tfaCode: twoFactorCode,
        };
        const data = await authClient.loginAccess(loginDetails, crypto_service_1.CryptoService.cryptoProvider);
        const { user, newToken } = data;
        const clearMnemonic = crypto_service_1.CryptoService.instance.decryptTextWithKey(user.mnemonic, password);
        const clearUser = {
            ...user,
            mnemonic: clearMnemonic,
        };
        return {
            user: clearUser,
            token: newToken,
        };
    };
    is2FANeeded = async (email) => {
        const authClient = sdk_manager_service_1.SdkManager.instance.getAuth();
        const securityDetails = await authClient.securityDetails(email).catch((error) => {
            throw new Error(error.message ?? 'Login error');
        });
        return securityDetails.tfaEnabled;
    };
    getAuthDetails = async () => {
        let loginCreds = await config_service_1.ConfigService.instance.readUser();
        if (!loginCreds?.token || !loginCreds?.user?.mnemonic) {
            throw new command_types_1.MissingCredentialsError();
        }
        const tokenDetails = validation_service_1.ValidationService.instance.validateTokenAndCheckExpiration(loginCreds.token);
        const isValidMnemonic = validation_service_1.ValidationService.instance.validateMnemonic(loginCreds.user.mnemonic);
        if (!tokenDetails.isValid || !isValidMnemonic) {
            throw new command_types_1.InvalidCredentialsError();
        }
        if (tokenDetails.expiration.expired) {
            throw new command_types_1.ExpiredCredentialsError();
        }
        const refreshToken = tokenDetails.expiration.refreshRequired;
        if (refreshToken) {
            loginCreds = await this.refreshUserToken(loginCreds.token, loginCreds.user.mnemonic);
        }
        return loginCreds;
    };
    refreshUserToken = async (oldToken, mnemonic) => {
        sdk_manager_service_1.SdkManager.init({ token: oldToken });
        const isValidMnemonic = validation_service_1.ValidationService.instance.validateMnemonic(mnemonic);
        if (!isValidMnemonic) {
            throw new command_types_1.InvalidCredentialsError();
        }
        const usersClient = sdk_manager_service_1.SdkManager.instance.getUsers();
        const newCreds = await usersClient.refreshUserCredentials();
        sdk_manager_service_1.SdkManager.init({ token: newCreds.newToken });
        const newLoginCreds = {
            user: {
                ...newCreds.user,
                mnemonic: mnemonic,
            },
            token: newCreds.newToken,
        };
        await config_service_1.ConfigService.instance.saveUser(newLoginCreds);
        return newLoginCreds;
    };
    logout = async () => {
        try {
            const user = await config_service_1.ConfigService.instance.readUser();
            if (!user || !user.token) {
                return;
            }
            const authClient = sdk_manager_service_1.SdkManager.instance.getAuth();
            return authClient.logout(user.token);
        }
        catch {
        }
    };
}
exports.AuthService = AuthService;
