"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const cli_utils_1 = require("../utils/cli.utils");
const config_service_1 = require("../services/config.service");
const validation_service_1 = require("../services/validation.service");
const auth_service_1 = require("../services/auth.service");
class Whoami extends core_1.Command {
    static args = {};
    static description = 'Display the current user logged into the Internxt CLI.';
    static aliases = [];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {};
    static enableJsonFlag = true;
    run = async () => {
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials) {
            const message = 'You are not logged in.';
            cli_utils_1.CLIUtils.error(this.log.bind(this), message);
            return { success: false, message };
        }
        else {
            const validCreds = this.checkUserAndTokens(userCredentials);
            if (!validCreds.valid) {
                const message = 'Your session has expired. You have been logged out. Please log in again.';
                await config_service_1.ConfigService.instance.clearUser();
                cli_utils_1.CLIUtils.error(this.log.bind(this), message);
                return { success: false, message };
            }
            else {
                if (validCreds.refreshRequired) {
                    try {
                        await auth_service_1.AuthService.instance.refreshUserToken(userCredentials.token, userCredentials.user.mnemonic);
                    }
                    catch {
                    }
                }
                const message = `You are logged in as: ${userCredentials.user.email}.`;
                cli_utils_1.CLIUtils.success(this.log.bind(this), message);
                return { success: true, message, login: userCredentials };
            }
        }
    };
    catch = async (error) => {
        const { flags } = await this.parse(Whoami);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    checkUserAndTokens = (loginCreds) => {
        if (!(loginCreds?.token && loginCreds?.user?.mnemonic)) {
            return { valid: false, refreshRequired: false };
        }
        const tokenDetails = validation_service_1.ValidationService.instance.validateTokenAndCheckExpiration(loginCreds.token);
        const goodMnemonic = validation_service_1.ValidationService.instance.validateMnemonic(loginCreds.user.mnemonic);
        const goodToken = tokenDetails.isValid && !tokenDetails.expiration.expired;
        return { valid: goodToken && goodMnemonic, refreshRequired: tokenDetails.expiration.refreshRequired };
    };
}
exports.default = Whoami;
