"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const pm2_utils_1 = require("../utils/pm2.utils");
const cli_utils_1 = require("../utils/cli.utils");
const config_service_1 = require("../services/config.service");
const auth_service_1 = require("../services/auth.service");
class Webdav extends core_1.Command {
    static args = {
        action: core_1.Args.string({
            required: true,
            options: ['enable', 'disable', 'restart', 'status'],
        }),
    };
    static description = 'Enable, disable, restart or get the status of the Internxt CLI WebDav server';
    static aliases = [];
    static examples = [
        '<%= config.bin %> <%= command.id %> enable',
        '<%= config.bin %> <%= command.id %> disable',
        '<%= config.bin %> <%= command.id %> restart',
        '<%= config.bin %> <%= command.id %> status',
    ];
    static flags = {};
    static enableJsonFlag = true;
    run = async () => {
        const { args, flags } = await this.parse(Webdav);
        let message = '';
        let success = true;
        await pm2_utils_1.PM2Utils.connect();
        switch (args.action) {
            case 'enable': {
                await auth_service_1.AuthService.instance.getAuthDetails();
                message = await this.enableWebDav(flags['json']);
                break;
            }
            case 'disable': {
                message = await this.disableWebDav(flags['json']);
                break;
            }
            case 'restart': {
                await auth_service_1.AuthService.instance.getAuthDetails();
                message = await this.restartWebDav(flags['json']);
                break;
            }
            case 'status': {
                await auth_service_1.AuthService.instance.getAuthDetails();
                message = await this.webDAVStatus();
                break;
            }
            default: {
                success = false;
                message = `Expected one of this command actions: ${Webdav.args.action.options}`;
                break;
            }
        }
        pm2_utils_1.PM2Utils.disconnect();
        return { success, message, action: args.action };
    };
    catch = async (error) => {
        const { flags } = await this.parse(Webdav);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    enableWebDav = async (jsonFlag) => {
        cli_utils_1.CLIUtils.doing('Starting Internxt WebDav server...', jsonFlag);
        await pm2_utils_1.PM2Utils.killWebDavServer();
        await pm2_utils_1.PM2Utils.startWebDavServer();
        cli_utils_1.CLIUtils.done(jsonFlag);
        const { status } = await pm2_utils_1.PM2Utils.webdavServerStatus();
        const webdavConfigs = await config_service_1.ConfigService.instance.readWebdavConfig();
        if (status === 'online') {
            const message = 'Internxt WebDav server started successfully at ' +
                `${webdavConfigs.protocol}://${webdavConfigs.host}:${webdavConfigs.port}`;
            cli_utils_1.CLIUtils.log(this.log.bind(this), `\nWebDav server status: ${core_1.ux.colorize('green', 'online')}\n`);
            cli_utils_1.CLIUtils.success(this.log.bind(this), message);
            return message;
        }
        else {
            const message = `WebDav server status: ${core_1.ux.colorize('red', status)}`;
            cli_utils_1.CLIUtils.log(this.log.bind(this), message);
            return message;
        }
    };
    disableWebDav = async (jsonFlag) => {
        cli_utils_1.CLIUtils.doing('Stopping Internxt WebDav server...', jsonFlag);
        await pm2_utils_1.PM2Utils.killWebDavServer();
        cli_utils_1.CLIUtils.done(jsonFlag);
        const message = 'Internxt WebDav server stopped successfully';
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return message;
    };
    restartWebDav = async (jsonFlag) => {
        cli_utils_1.CLIUtils.doing('Restarting Internxt WebDav server...', jsonFlag);
        const { status } = await pm2_utils_1.PM2Utils.webdavServerStatus();
        if (status === 'online') {
            await pm2_utils_1.PM2Utils.killWebDavServer();
            await pm2_utils_1.PM2Utils.startWebDavServer();
            cli_utils_1.CLIUtils.done(jsonFlag);
            const message = 'Internxt WebDav server restarted successfully';
            cli_utils_1.CLIUtils.success(this.log.bind(this), message);
            return message;
        }
        else {
            cli_utils_1.CLIUtils.done(jsonFlag);
            const message = 'Internxt WebDav server is not running, it wont be restarted';
            cli_utils_1.CLIUtils.warning(this.log.bind(this), message);
            return message;
        }
    };
    webDAVStatus = async () => {
        const { status } = await pm2_utils_1.PM2Utils.webdavServerStatus();
        const message = `Internxt WebDAV server status: ${status}`;
        cli_utils_1.CLIUtils.log(this.log.bind(this), message);
        return message;
    };
}
exports.default = Webdav;
