"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const cli_utils_1 = require("../utils/cli.utils");
const auth_service_1 = require("../services/auth.service");
const validation_service_1 = require("../services/validation.service");
const config_service_1 = require("../services/config.service");
const upload_facade_service_1 = require("../services/network/upload/upload-facade.service");
const command_types_1 = require("../types/command.types");
class UploadFolder extends core_1.Command {
    static args = {};
    static description = 'Upload a folder to Internxt Drive';
    static aliases = ['upload:folder'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        folder: core_1.Flags.string({
            char: 'f',
            description: 'The path to the folder on your system.',
            required: false,
        }),
        destination: core_1.Flags.string({
            char: 'i',
            description: 'The folder id where the folder is going to be uploaded to. Leave empty for the root folder.',
            required: false,
            parse: cli_utils_1.CLIUtils.parseEmpty,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { user } = await auth_service_1.AuthService.instance.getAuthDetails();
        const { flags } = await this.parse(UploadFolder);
        const localPath = await this.getFolderPath(flags['folder'], flags['non-interactive']);
        const destinationFolderUuid = (await cli_utils_1.CLIUtils.getDestinationFolderUuid({
            destinationFolderUuidFlag: flags['destination'],
            destinationFlagName: UploadFolder.flags['destination'].name,
            nonInteractive: flags['non-interactive'],
            reporter: this.log.bind(this),
        })) ?? user.rootFolderId;
        const progressBar = cli_utils_1.CLIUtils.progress({
            format: 'Uploading folder [{bar}] {percentage}%',
            linewrap: true,
        }, flags['json']);
        progressBar?.start(100, 0);
        const data = await upload_facade_service_1.UploadFacade.instance.uploadFolder({
            localPath,
            destinationFolderUuid,
            loginUserDetails: user,
            jsonFlag: flags['json'],
            onProgress: (progress) => {
                progressBar?.update(progress.percentage);
            },
        });
        progressBar?.update(100);
        progressBar?.stop();
        const driveUrl = config_service_1.ConfigService.instance.get('DRIVE_WEB_URL');
        const folderUrl = `${driveUrl}/folder/${data.rootFolderId}`;
        const message = `Folder uploaded in ${data.uploadTimeMs}ms, view it at ${folderUrl} (${data.totalBytes} bytes)`;
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
    };
    catch = async (error) => {
        const { flags } = await this.parse(UploadFolder);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFolderPath = async (folderFlag, nonInteractive) => {
        return await cli_utils_1.CLIUtils.getValueFromFlag({
            value: folderFlag,
            name: UploadFolder.flags['folder'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the path to the folder on your computer?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateDirectoryExists,
            error: new command_types_1.NotValidDirectoryError(),
        }, this.log.bind(this));
    };
}
exports.default = UploadFolder;
