"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const promises_1 = require("node:fs/promises");
const node_fs_1 = require("node:fs");
const auth_service_1 = require("../services/auth.service");
const cli_utils_1 = require("../utils/cli.utils");
const config_service_1 = require("../services/config.service");
const node_path_1 = __importDefault(require("node:path"));
const drive_file_service_1 = require("../services/drive/drive-file.service");
const errors_utils_1 = require("../utils/errors.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
const types_1 = require("@internxt/sdk/dist/drive/storage/types");
const thumbnail_service_1 = require("../services/thumbnail.service");
const stream_utils_1 = require("../utils/stream.utils");
const thumbnail_utils_1 = require("../utils/thumbnail.utils");
class UploadFile extends core_1.Command {
    static args = {};
    static description = 'Upload a file to Internxt Drive';
    static aliases = ['upload:file'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        file: core_1.Flags.string({
            char: 'f',
            description: 'The path to the file on your system.',
            required: false,
        }),
        destination: core_1.Flags.string({
            char: 'i',
            description: 'The folder id where the file is going to be uploaded to. Leave empty for the root folder.',
            required: false,
            parse: cli_utils_1.CLIUtils.parseEmpty,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(UploadFile);
        const nonInteractive = flags['non-interactive'];
        const { user } = await auth_service_1.AuthService.instance.getAuthDetails();
        const filePath = await this.getFilePath(flags['file'], nonInteractive);
        const stats = await (0, promises_1.stat)(filePath);
        if (!stats.size) {
            throw new Error('The file is empty. Uploading empty files is not allowed.');
        }
        const fileInfo = node_path_1.default.parse(filePath);
        const fileType = fileInfo.ext.replaceAll('.', '');
        const destinationFolderUuid = (await cli_utils_1.CLIUtils.getDestinationFolderUuid({
            destinationFolderUuidFlag: flags['destination'],
            destinationFlagName: UploadFile.flags['destination'].name,
            nonInteractive,
            reporter: this.log.bind(this),
        })) ?? user.rootFolderId;
        const networkFacade = await cli_utils_1.CLIUtils.prepareNetwork({ loginUserDetails: user, jsonFlag: flags['json'] });
        const readStream = (0, node_fs_1.createReadStream)(filePath);
        const timer = cli_utils_1.CLIUtils.timer();
        const progressBar = cli_utils_1.CLIUtils.progress({
            format: 'Uploading file [{bar}] {percentage}%',
            linewrap: true,
        }, flags['json']);
        progressBar?.start(100, 0);
        let bufferStream;
        let fileStream = readStream;
        const isThumbnailable = (0, thumbnail_utils_1.isFileThumbnailable)(fileType);
        if (isThumbnailable) {
            bufferStream = new stream_utils_1.BufferStream();
            fileStream = readStream.pipe(bufferStream);
        }
        const progressCallback = (progress) => {
            progressBar?.update(progress * 100 * 0.99);
        };
        const fileId = await new Promise((resolve, reject) => {
            const state = networkFacade.uploadFile(fileStream, stats.size, user.bucket, (err, res) => {
                if (err) {
                    return reject(err);
                }
                resolve(res);
            }, progressCallback);
            process.on('SIGINT', () => {
                state.stop();
                process.exit(1);
            });
        });
        const createdDriveFile = await drive_file_service_1.DriveFileService.instance.createFile({
            plainName: fileInfo.name,
            type: fileType,
            size: stats.size,
            folderUuid: destinationFolderUuid,
            fileId: fileId,
            bucket: user.bucket,
            encryptVersion: types_1.EncryptionVersion.Aes03,
            creationTime: stats.birthtime?.toISOString(),
            modificationTime: stats.mtime?.toISOString(),
        });
        try {
            if (isThumbnailable && bufferStream) {
                const thumbnailBuffer = bufferStream.getBuffer();
                if (thumbnailBuffer) {
                    await thumbnail_service_1.ThumbnailService.instance.uploadThumbnail(thumbnailBuffer, fileType, user.bucket, createdDriveFile.uuid, networkFacade);
                }
            }
        }
        catch (error) {
            errors_utils_1.ErrorUtils.report(error, { command: this.id });
        }
        progressBar?.update(100);
        progressBar?.stop();
        const uploadTime = timer.stop();
        this.log('\n');
        const message = `File uploaded in ${uploadTime}ms, view it at ${config_service_1.ConfigService.instance.get('DRIVE_WEB_URL')}/file/${createdDriveFile.uuid}`;
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return {
            success: true,
            message,
            file: {
                ...createdDriveFile,
                plainName: fileInfo.name,
            },
        };
    };
    catch = async (error) => {
        const { flags } = await this.parse(UploadFile);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFilePath = async (fileFlag, nonInteractive) => {
        const filePath = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: fileFlag,
            name: UploadFile.flags['file'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the path to the file on your computer?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateFileExists,
            error: new command_types_1.NotValidDirectoryError(),
        }, this.log.bind(this));
        return filePath;
    };
}
exports.default = UploadFile;
