"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
const trash_service_1 = require("../services/drive/trash.service");
class TrashFile extends core_1.Command {
    static args = {};
    static description = 'Moves a given file to the trash.';
    static aliases = ['trash:file'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        id: core_1.Flags.string({
            char: 'i',
            description: 'The file id to be trashed.',
            required: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(TrashFile);
        const nonInteractive = flags['non-interactive'];
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        const uuid = await this.getFileUuid(flags['id'], nonInteractive);
        await trash_service_1.TrashService.instance.trashItems({ items: [{ uuid, type: 'file' }] });
        const message = 'File trashed successfully.';
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message, file: { uuid } };
    };
    catch = async (error) => {
        const { flags } = await this.parse(TrashFile);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFileUuid = async (fileUuidFlag, nonInteractive) => {
        const fileUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: fileUuidFlag,
            name: TrashFile.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the file id you want to trash?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFileUuidError(),
        }, this.log.bind(this));
        return fileUuid;
    };
}
exports.default = TrashFile;
