"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const trash_service_1 = require("../services/drive/trash.service");
const inquirer_utils_1 = require("../utils/inquirer.utils");
class TrashClear extends core_1.Command {
    static args = {};
    static description = 'Deletes permanently all the content of the trash. This action cannot be undone.';
    static aliases = ['trash:clear'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        force: core_1.Flags.boolean({
            char: 'f',
            description: 'It forces the trash to be emptied without confirmation.',
            required: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(TrashClear);
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        if (!flags.force) {
            if (flags['non-interactive']) {
                const message = 'The "non interactive" flag is enabled, but the "force" flag has not been provided. User confirmation is required to empty the trash permanently.';
                cli_utils_1.CLIUtils.error(this.log.bind(this), message);
                return { success: false, message };
            }
            const confirmation = await this.getConfirmation();
            if (confirmation !== 'y') {
                const message = 'User confirmation is required to empty the trash permanently.';
                cli_utils_1.CLIUtils.error(this.log.bind(this), message);
                return { success: false, message };
            }
        }
        await trash_service_1.TrashService.instance.clearTrash();
        const message = 'Trash emptied successfully.';
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message };
    };
    catch = async (error) => {
        const { flags } = await this.parse(TrashClear);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getConfirmation = async () => {
        let confirmation = (await this.getConfirmationInteractively()).trim().toLowerCase();
        if (confirmation.length === 0) {
            confirmation = 'no';
        }
        return confirmation.charAt(0);
    };
    getConfirmationInteractively = () => {
        return inquirer_utils_1.InquirerUtils.prompt('Empty trash? All items in the Drive Trash will be permanently deleted. This action cannot be undone.', {
            type: 'confirm',
            confirm: { default: false },
        });
    };
}
exports.default = TrashClear;
