"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const drive_folder_service_1 = require("../services/drive/drive-folder.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
const format_utils_1 = require("../utils/format.utils");
class List extends core_1.Command {
    static args = {};
    static description = 'Lists the content of a folder id.';
    static aliases = [];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        id: core_1.Flags.string({
            char: 'i',
            description: 'The folder id to list. Leave empty for the root folder.',
            required: false,
            parse: cli_utils_1.CLIUtils.parseEmpty,
        }),
        extended: core_1.Flags.boolean({
            char: 'e',
            description: 'Displays additional information in the list.',
            required: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(List);
        const nonInteractive = flags['non-interactive'];
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        let folderUuid = await this.getFolderUuid(flags['id'], nonInteractive);
        if (folderUuid.trim().length === 0) {
            folderUuid = userCredentials.user.rootFolderId;
        }
        const { folders, files } = await drive_folder_service_1.DriveFolderService.instance.getFolderContent(folderUuid);
        const allItems = [
            ...folders.map((folder) => {
                return {
                    type: 'folder',
                    name: folder.plainName,
                    id: folder.uuid,
                    size: '-',
                    modified: format_utils_1.FormatUtils.formatDate(folder.updatedAt),
                };
            }),
            ...files.map((file) => {
                return {
                    type: 'file',
                    name: file.type && file.type.length > 0 ? `${file.plainName}.${file.type}` : file.plainName,
                    id: file.uuid,
                    size: format_utils_1.FormatUtils.humanFileSize(Number(file.size)),
                    modified: format_utils_1.FormatUtils.formatDate(file.updatedAt),
                };
            }),
        ];
        const headers = [
            { value: 'type', alias: 'Type' },
            { value: 'name', alias: 'Name' },
            { value: 'id', alias: 'Id' },
        ];
        if (flags.extended) {
            headers.push({ value: 'modified', alias: 'Modified' }, { value: 'size', alias: 'Size' });
        }
        cli_utils_1.CLIUtils.table(this.log.bind(this), headers, allItems);
        return { success: true, list: { folders, files } };
    };
    catch = async (error) => {
        const { flags } = await this.parse(List);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFolderUuid = async (folderUuidFlag, nonInteractive) => {
        const folderUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: folderUuidFlag,
            name: List.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the folder id you want to list? (leave empty for the root folder)',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFolderUuidError(),
            canBeEmpty: true,
        }, this.log.bind(this));
        return folderUuid;
    };
}
exports.default = List;
