"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const drive_file_service_1 = require("../services/drive/drive-file.service");
const cli_utils_1 = require("../utils/cli.utils");
const auth_service_1 = require("../services/auth.service");
const node_fs_1 = require("node:fs");
const promises_1 = __importDefault(require("node:fs/promises"));
const node_path_1 = __importDefault(require("node:path"));
const stream_utils_1 = require("../utils/stream.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
class DownloadFile extends core_1.Command {
    static args = {};
    static description = 'Download and decrypts a file from Internxt Drive to a directory.' +
        ' The file name will be the same as the file name in your Drive.';
    static aliases = ['download:file'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        id: core_1.Flags.string({
            char: 'i',
            description: 'The id of the file to download. Use <%= config.bin %> list to view your files ids',
            required: false,
        }),
        directory: core_1.Flags.string({
            char: 'd',
            description: 'The directory to download the file to. Leave empty for the current folder.',
            required: false,
            parse: cli_utils_1.CLIUtils.parseEmpty,
        }),
        overwrite: core_1.Flags.boolean({
            char: 'o',
            description: 'Overwrite the file if it already exists',
            default: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(DownloadFile);
        const nonInteractive = flags['non-interactive'];
        const overwrite = flags['overwrite'];
        let downloadDirectory = await this.getDirectory(flags['directory'], nonInteractive);
        if (downloadDirectory.trim().length === 0) {
            downloadDirectory = '.';
        }
        const fileUuid = await this.getFileUuid(flags['id'], nonInteractive);
        const driveFile = await this.getFileMetadata(fileUuid, flags['json']);
        const downloadPath = await this.getDownloadPath(downloadDirectory, driveFile, overwrite);
        const progressBar = cli_utils_1.CLIUtils.progress({
            format: 'Downloading file [{bar}] {percentage}%',
            linewrap: true,
        }, flags['json']);
        progressBar?.start(100, 0);
        if (driveFile.size === 0) {
            await promises_1.default.writeFile(downloadPath, '');
        }
        else {
            if (!driveFile.fileId) {
                throw new command_types_1.NotValidFileIdError();
            }
            const { user } = await auth_service_1.AuthService.instance.getAuthDetails();
            const networkFacade = cli_utils_1.CLIUtils.prepareNetwork({ loginUserDetails: user, jsonFlag: flags['json'] });
            const fileWriteStream = (0, node_fs_1.createWriteStream)(downloadPath);
            const [executeDownload, abortable] = await networkFacade.downloadToStream(driveFile.bucket, user.mnemonic, driveFile.fileId, driveFile.size, stream_utils_1.StreamUtils.writeStreamToWritableStream(fileWriteStream), undefined, {
                abortController: new AbortController(),
                progressCallback: (progress) => {
                    progressBar?.update(progress * 0.99);
                },
            });
            process.on('SIGINT', () => {
                abortable.abort('SIGINT received');
                process.exit(1);
            });
            await executeDownload;
        }
        try {
            await promises_1.default.utimes(downloadPath, new Date(), driveFile.modificationTime ?? driveFile.updatedAt);
        }
        catch {
        }
        progressBar?.update(100);
        progressBar?.stop();
        const message = `File downloaded successfully to ${downloadPath}`;
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message, file: driveFile };
    };
    catch = async (error) => {
        const { flags } = await this.parse(DownloadFile);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFileUuid = async (fileUuidFlag, nonInteractive) => {
        const fileUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: fileUuidFlag,
            name: DownloadFile.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the file id you want to download?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFileUuidError(),
        }, this.log.bind(this));
        return fileUuid;
    };
    getDirectory = async (directoryFlag, nonInteractive) => {
        const directory = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: directoryFlag,
            name: DownloadFile.flags['directory'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'Where would you like to download the file? (Enter the local folder path on your computer)',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateDirectoryExists,
            error: new command_types_1.NotValidDirectoryError(),
            canBeEmpty: true,
        }, this.log.bind(this));
        return directory;
    };
    getFileMetadata = async (uuid, jsonFlag) => {
        cli_utils_1.CLIUtils.doing('Getting file metadata', jsonFlag);
        const driveFile = await drive_file_service_1.DriveFileService.instance.getFileMetadata(uuid);
        cli_utils_1.CLIUtils.done(jsonFlag);
        if (!driveFile) {
            throw new Error('File not found');
        }
        return driveFile;
    };
    getDownloadPath = async (downloadDirectory, driveFile, overwrite) => {
        const ext = driveFile.type?.length && driveFile.type.length > 0 ? `.${driveFile.type}` : undefined;
        const filename = node_path_1.default.format({
            name: driveFile.name,
            ext: ext,
        });
        const downloadPath = node_path_1.default.join(downloadDirectory, filename);
        await promises_1.default.access(downloadDirectory, promises_1.default.constants.W_OK);
        try {
            const stat = await promises_1.default.stat(downloadPath);
            if (stat.isFile() && !overwrite) {
                throw new Error('File already exists, use --overwrite flag to overwrite it');
            }
        }
        catch (err) {
            if (err.code !== 'ENOENT') {
                throw err;
            }
        }
        return downloadPath;
    };
}
exports.default = DownloadFile;
