"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
const trash_service_1 = require("../services/drive/trash.service");
const drive_folder_service_1 = require("../services/drive/drive-folder.service");
class DeletePermanentlyFolder extends core_1.Command {
    static args = {};
    static description = 'Deletes permanently a folder. This action cannot be undone.';
    static aliases = ['delete:permanently:folder'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        id: core_1.Flags.string({
            char: 'i',
            description: 'The folder id to be permanently deleted.',
            required: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(DeletePermanentlyFolder);
        const nonInteractive = flags['non-interactive'];
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        const folderUuid = await this.getFolderUuid(flags['id'], nonInteractive);
        const driveFolder = await drive_folder_service_1.DriveFolderService.instance.getFolderMetaByUuid(folderUuid);
        if (!driveFolder) {
            throw new Error('Folder not found');
        }
        await trash_service_1.TrashService.instance.deleteFolder(driveFolder.uuid);
        const message = 'Folder permanently deleted successfully';
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message };
    };
    catch = async (error) => {
        const { flags } = await this.parse(DeletePermanentlyFolder);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFolderUuid = async (folderUuidFlag, nonInteractive) => {
        const folderUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: folderUuidFlag,
            name: DeletePermanentlyFolder.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the folder id you want to permanently delete? (This action cannot be undone)',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFolderUuidError(),
        }, this.log.bind(this));
        return folderUuid;
    };
}
exports.default = DeletePermanentlyFolder;
