"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
const drive_file_service_1 = require("../services/drive/drive-file.service");
const trash_service_1 = require("../services/drive/trash.service");
class DeletePermanentlyFile extends core_1.Command {
    static args = {};
    static description = 'Deletes permanently a file. This action cannot be undone.';
    static aliases = ['delete:permanently:file'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        id: core_1.Flags.string({
            char: 'i',
            description: 'The file id to be permanently deleted.',
            required: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(DeletePermanentlyFile);
        const nonInteractive = flags['non-interactive'];
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        const fileUuid = await this.getFileUuid(flags['id'], nonInteractive);
        const driveFile = await drive_file_service_1.DriveFileService.instance.getFileMetadata(fileUuid);
        if (!driveFile) {
            throw new Error('File not found');
        }
        await trash_service_1.TrashService.instance.deleteFile(driveFile.uuid);
        const message = 'File permanently deleted successfully';
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message };
    };
    catch = async (error) => {
        const { flags } = await this.parse(DeletePermanentlyFile);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFileUuid = async (fileUuidFlag, nonInteractive) => {
        const fileUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: fileUuidFlag,
            name: DeletePermanentlyFile.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the file id you want to permanently delete? (This action cannot be undone)',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFileUuidError(),
        }, this.log.bind(this));
        return fileUuid;
    };
}
exports.default = DeletePermanentlyFile;
