"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const cli_utils_1 = require("../utils/cli.utils");
const drive_folder_service_1 = require("../services/drive/drive-folder.service");
const config_service_1 = require("../services/config.service");
const validation_service_1 = require("../services/validation.service");
const command_types_1 = require("../types/command.types");
const async_utils_1 = require("../utils/async.utils");
class CreateFolder extends core_1.Command {
    static args = {};
    static description = 'Create a folder in your Internxt Drive';
    static aliases = [];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        name: core_1.Flags.string({
            char: 'n',
            description: 'The new name for the folder',
            required: false,
        }),
        id: core_1.Flags.string({
            char: 'i',
            description: 'The ID of the folder where the new folder will be created. Defaults to your root folder if not specified.',
            required: false,
            parse: cli_utils_1.CLIUtils.parseEmpty,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(CreateFolder);
        const nonInteractive = flags['non-interactive'];
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        const folderName = await this.getFolderName(flags['name'], nonInteractive);
        let folderUuid = await this.getFolderUuid(flags['id'], nonInteractive);
        if (folderUuid.trim().length === 0) {
            folderUuid = userCredentials.user.rootFolderId;
        }
        cli_utils_1.CLIUtils.doing('Creating folder...', flags['json']);
        const [createNewFolder, requestCanceler] = drive_folder_service_1.DriveFolderService.instance.createFolder({
            plainName: folderName,
            parentFolderUuid: folderUuid,
        });
        process.on('SIGINT', () => {
            requestCanceler.cancel('SIGINT received');
            process.exit(1);
        });
        const newFolder = await createNewFolder;
        await async_utils_1.AsyncUtils.sleep(500);
        cli_utils_1.CLIUtils.done(flags['json']);
        const message = `Folder ${newFolder.plainName} created successfully, view it at ${config_service_1.ConfigService.instance.get('DRIVE_WEB_URL')}/folder/${newFolder.uuid}`;
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message, folder: newFolder };
    };
    catch = async (error) => {
        const { flags } = await this.parse(CreateFolder);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFolderName = async (folderNameFlag, nonInteractive) => {
        const folderName = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: folderNameFlag,
            name: CreateFolder.flags['name'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What would you like to name the new folder?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateStringIsNotEmpty,
            error: new command_types_1.EmptyFolderNameError(),
        }, this.log.bind(this));
        return folderName;
    };
    getFolderUuid = async (folderUuidFlag, nonInteractive) => {
        const folderUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: folderUuidFlag,
            name: CreateFolder.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the ID of the folder where you would like to create the new folder? (leave empty for the root folder)',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFolderUuidError(),
            canBeEmpty: true,
        }, this.log.bind(this));
        return folderUuid;
    };
}
exports.default = CreateFolder;
