"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpClient = void 0;
var axios_1 = __importDefault(require("axios"));
var errors_1 = __importDefault(require("../types/errors"));
var HttpClient = /** @class */ (function () {
    function HttpClient(baseURL, unauthorizedCallback) {
        this.axios = axios_1.default.create({
            baseURL: baseURL,
        });
        this.unauthorizedCallback = unauthorizedCallback;
        this.initializeMiddleware();
    }
    HttpClient.create = function (baseURL, unauthorizedCallback) {
        if (unauthorizedCallback === undefined) {
            unauthorizedCallback = function () { return null; };
        }
        return new HttpClient(baseURL, unauthorizedCallback);
    };
    /**
     * Requests a GET
     * @param url
     * @param headers
     */
    HttpClient.prototype.get = function (url, headers) {
        return this.axios.get(url, {
            headers: headers,
        });
    };
    /**
     * Requests a GET
     * @param url
     * @param params
     * @param headers
     */
    HttpClient.prototype.getWithParams = function (url, params, headers) {
        return this.axios.get(url, {
            params: params,
            headers: headers,
        });
    };
    /**
     * Requests a GET with option to cancel
     * @param url
     * @param headers
     */
    HttpClient.prototype.getCancellable = function (url, headers) {
        var cancelTokenSource = axios_1.default.CancelToken.source();
        var config = {
            headers: headers,
            cancelToken: cancelTokenSource.token,
        };
        var promise = this.axios.get(url, config);
        return {
            promise: promise,
            requestCanceler: {
                cancel: cancelTokenSource.cancel,
            },
        };
    };
    /**
     * Requests a POST
     * @param url
     * @param params
     * @param headers
     */
    HttpClient.prototype.post = function (url, params, headers) {
        return this.axios.post(url, params, {
            headers: headers,
        });
    };
    /**
     * Requests a POST FORM
     * @param url
     * @param params
     * @param headers
     */
    HttpClient.prototype.postForm = function (url, params, headers) {
        return this.axios.postForm(url, params, {
            headers: headers,
        });
    };
    /**
     * Requests a POST with option to cancel
     * @param url
     * @param params
     * @param headers
     */
    HttpClient.prototype.postCancellable = function (url, params, headers) {
        var cancelTokenSource = axios_1.default.CancelToken.source();
        var config = {
            headers: headers,
            cancelToken: cancelTokenSource.token,
        };
        var promise = this.axios.post(url, params, config);
        return {
            promise: promise,
            requestCanceler: {
                cancel: cancelTokenSource.cancel,
            },
        };
    };
    /**
     * Requests PATCH
     * @param url
     * @param params
     * @param headers
     */
    HttpClient.prototype.patch = function (url, params, headers) {
        return this.axios.patch(url, params, {
            headers: headers,
        });
    };
    /**
     * Requests a PUT
     * @param url
     * @param params
     * @param headers
     */
    HttpClient.prototype.put = function (url, params, headers) {
        return this.axios.put(url, params, {
            headers: headers,
        });
    };
    /**
     * Requests a PUT FORM
     * @param url
     * @param params
     * @param headers
     */
    HttpClient.prototype.putForm = function (url, params, headers) {
        return this.axios.putForm(url, params, {
            headers: headers,
        });
    };
    /**
     * Requests a DELETE
     * @param url
     * @param headers
     * @param params
     */
    HttpClient.prototype.delete = function (url, headers, params) {
        return this.axios.delete(url, {
            headers: headers,
            data: params,
        });
    };
    /**
     * Sets middlewares into the client for common behaviour managing
     * @private
     */
    HttpClient.prototype.initializeMiddleware = function () {
        this.axios.interceptors.response.use(HttpClient.extractData, this.normalizeError.bind(this));
    };
    /**
     * Extracts the valuable data from the server's response
     * @param response
     * @private
     */
    HttpClient.extractData = function (response) {
        return response.data;
    };
    /**
     * Converts the specific client error into a proprietary error for our apps
     * @param error
     * @private
     */
    HttpClient.prototype.normalizeError = function (error) {
        var errorMessage, errorStatus, errorCode;
        if (error.response) {
            var response = error.response;
            if (response.status === 401) {
                this.unauthorizedCallback();
            }
            errorMessage = response.data.message || response.data.error || JSON.stringify(response.data);
            errorStatus = response.status;
            errorCode = response.data.code;
        }
        else if (error.request) {
            errorMessage = 'Server unavailable';
            errorStatus = 500;
        }
        else {
            errorMessage = error.message;
            errorStatus = 400;
        }
        throw new errors_1.default(errorMessage, errorStatus, errorCode);
    };
    return HttpClient;
}());
exports.HttpClient = HttpClient;
