"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var crypto_1 = require("crypto");
var buffer_1 = require("buffer");
var uuid_1 = require("uuid");
/**
 * Converts a standard Base64 string into a URL-safe Base64 variant:
 *  - Replaces all "+" characters with "-"
 *  - Replaces all "/" characters with "_"
 *  - Strips any trailing "=" padding characters
 *
 * @param base64 - A standard Base64–encoded string
 * @returns The URL-safe Base64 string
 */
var toBase64UrlSafe = function (base64) {
    return base64
        .replace(/\+/g, '-') // converts "+" to "-"
        .replace(/\//g, '_') // converts "/" to "_"
        .replace(/=+$/, ''); // removes trailing "="
};
/**
 * Converts a URL-safe Base64 string back into a standard Base64 variant:
 *  - Replaces all "-" characters with "+"
 *  - Replaces all "_" characters with "/"
 *  - Adds "=" padding characters at the end until length is a multiple of 4
 *
 * @param urlSafe - A URL-safe Base64–encoded string
 * @returns The standard Base64 string, including any necessary "=" padding
 */
var fromBase64UrlSafe = function (urlSafe) {
    var base64 = urlSafe
        .replace(/-/g, '+') // convert "-" back to "+"
        .replace(/_/g, '/'); // convert "_" back to "/"
    var missingPadding = (4 - (base64.length % 4)) % 4;
    if (missingPadding > 0) {
        base64 += '='.repeat(missingPadding);
    }
    return base64;
};
/**
 * Generates a cryptographically secure, URL-safe string of a given length.
 *
 * Internally:
 * 1. Calculates how many raw bytes are needed to generate at least `size` Base64 chars.
 * 2. Generates secure random bytes with `crypto.randomBytes()`.
 * 3. Encodes to standard Base64, then makes it URL-safe.
 * 4. Truncates the result to `size` characters.
 *
 * @param size - Desired length of the output string (must be ≥1)
 * @returns A URL-safe string exactly `size` characters long
 */
var generateRandomStringUrlSafe = function (size) {
    if (size <= 0) {
        throw new Error('Size must be a positive integer');
    }
    // Base64 yields 4 chars per 3 bytes, so it computes the minimum bytes required
    var numBytes = Math.ceil((size * 3) / 4);
    var buf = (0, crypto_1.randomBytes)(numBytes).toString('base64');
    return toBase64UrlSafe(buf).substring(0, size);
};
/**
 * Converts a base64 url safe string to uuid v4
 *
 * @example in: `8yqR2seZThOqF4xNngMjyQ` out: `f32a91da-c799-4e13-aa17-8c4d9e0323c9`
 */
function base64UrlSafetoUUID(base64UrlSafe) {
    var hex = buffer_1.Buffer.from(fromBase64UrlSafe(base64UrlSafe), 'base64').toString('hex');
    return "".concat(hex.substring(0, 8), "-").concat(hex.substring(8, 12), "-").concat(hex.substring(12, 16)) +
        "-".concat(hex.substring(16, 20), "-").concat(hex.substring(20));
}
/**
 * Extracts a UUIDv4 from the provided parameter. If the value is not a valid UUIDv4, it tries to convert
 * the parameter from a Base64 URL-safe string to a UUID. In either case, it returns the provided value.
 *
 * @value value - The input parameter, which can be either a UUIDv4 or a Base64 URL-safe string.
 * @returns The decoded v4 UUID string.
 */
var decodeV4Uuid = function (value) {
    if (!(0, uuid_1.validate)(value)) {
        try {
            return base64UrlSafetoUUID(value);
        }
        catch (_a) {
            return value;
        }
    }
    return value;
};
/**
 * Encodes a UUIDv4 by removing hyphens, converting it from hexadecimal to Base64,
 * and then making it URL-safe.
 *
 * @param v4Uuid - The ID to be encoded, expected to be a UUIDv4 string.
 * @returns The encoded send ID as a URL-safe Base64 string.
 * @throws {Error} If the provided UUIDv4 is not valid.
 */
var encodeV4Uuid = function (v4Uuid) {
    if (!(0, uuid_1.validate)(v4Uuid)) {
        throw new Error('The provided UUIDv4 is not valid');
    }
    var removedUuidDecoration = v4Uuid.replace(/-/g, '');
    var base64endoded = buffer_1.Buffer.from(removedUuidDecoration, 'hex').toString('base64');
    var encodedSendId = toBase64UrlSafe(base64endoded);
    return encodedSendId;
};
exports.default = {
    toBase64UrlSafe: toBase64UrlSafe,
    fromBase64UrlSafe: fromBase64UrlSafe,
    generateRandomStringUrlSafe: generateRandomStringUrlSafe,
    base64UrlSafetoUUID: base64UrlSafetoUUID,
    decodeV4Uuid: decodeV4Uuid,
    encodeV4Uuid: encodeV4Uuid,
};
