"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Checkout = void 0;
var headers_1 = require("../shared/headers");
var client_1 = require("../shared/http/client");
var Checkout = /** @class */ (function () {
    function Checkout(apiUrl, appDetails, apiSecurity) {
        this.client = client_1.HttpClient.create(apiUrl, apiSecurity.unauthorizedCallback);
        this.appDetails = appDetails;
        this.apiSecurity = apiSecurity;
    }
    Checkout.client = function (apiUrl, appDetails, apiSecurity) {
        return new Checkout(apiUrl, appDetails, apiSecurity);
    };
    /**
     * @description Creates a customer or gets the existing one if it already exists
     * @param country - The country of the customer
     * @param postalCode - The postal code of the customer
     * @param companyVatId - The VAT ID of the company (optional)
     * @returns The customer ID and the user token used to create a subscription or payment intent
     */
    Checkout.prototype.getCustomerId = function (_a) {
        var customerName = _a.customerName, postalCode = _a.postalCode, country = _a.country, captchaToken = _a.captchaToken, companyVatId = _a.companyVatId;
        var query = new URLSearchParams();
        query.set('customerName', customerName);
        query.set('country', country);
        query.set('postalCode', postalCode);
        query.set('captchaToken', captchaToken);
        if (companyVatId !== undefined)
            query.set('companyVatId', companyVatId);
        return this.client.get("/checkout/customer?".concat(query.toString()), this.authHeaders());
    };
    /**
     * @description Creates a subscription for a given customer
     * @param customerId - The ID of the customer
     * @param priceId - The ID of the price
     * @param token - The token used to authenticate the customer
     * @param currency - The currency of the subscription (optional)
     * @param promoCodeId - The ID of the promo code (optional)
     * @param quantity - The quantity of the subscription (optional)
     * @returns The created subscription data:
     * - `type`: The type of the subscription (setup or payment)
     * - `clientSecret`: The client secret for the subscription to be used with Stripe Elements
     * - `subscriptionId`: The ID of the subscription (optional)
     * - `paymentIntentId`: The ID of the payment intent (optional)
     */
    Checkout.prototype.createSubscription = function (_a) {
        var customerId = _a.customerId, priceId = _a.priceId, token = _a.token, currency = _a.currency, captchaToken = _a.captchaToken, promoCodeId = _a.promoCodeId, quantity = _a.quantity;
        return this.client.post('/checkout/subscription', {
            customerId: customerId,
            priceId: priceId,
            token: token,
            currency: currency,
            captchaToken: captchaToken,
            promoCodeId: promoCodeId,
            quantity: quantity,
        }, this.authHeaders());
    };
    /**
     * @description Creates a payment intent for a given customer
     * @param customerId - The ID of the customer
     * @param priceId - The ID of the price
     * @param token - The token used to authenticate the customer
     * @param currency - The currency of the payment intent (optional)
     * @param promoCodeId - The ID of the promo code (optional)
     * @returns The created invoice data:
     * - `clientSecret`: The client secret for the invoice to be used with Stripe Elements
     * - `id`: The ID of the invoice
     * - `invoiceStatus`: The status of the invoice (only when the status is 'paid')
     * - `type`: The type of the currency - 'fiat' or 'crypto'
     * - `payload`: The payload of the invoice if the type is 'crypto' containing { paymentRequestUri, url, qrUrl }
     * - `payload.paymentRequestUri`: The payment request URI for the invoice
     * - `payload.url`: The URL of the invoice
     * - `payload.qrUrl`: The QR code URL of the invoice
     */
    Checkout.prototype.createPaymentIntent = function (_a) {
        var customerId = _a.customerId, priceId = _a.priceId, token = _a.token, currency = _a.currency, captchaToken = _a.captchaToken, promoCodeId = _a.promoCodeId;
        return this.client.post('/checkout/payment-intent', {
            customerId: customerId,
            priceId: priceId,
            token: token,
            currency: currency,
            captchaToken: captchaToken,
            promoCodeId: promoCodeId,
        }, this.authHeaders());
    };
    /**
     * @description Fetch a requested price by its ID and its tax rate
     * @param priceId - The ID of the price
     * @param promoCodeName - The name of the promo code (optional)
     * @param currency - The currency of the price (optional)
     * @returns The price object containing the details of the requested price
     */
    Checkout.prototype.getPriceById = function (_a) {
        var priceId = _a.priceId, promoCodeName = _a.promoCodeName, userAddress = _a.userAddress, currency = _a.currency, postalCode = _a.postalCode, country = _a.country;
        var query = new URLSearchParams();
        query.set('priceId', priceId);
        if (promoCodeName !== undefined)
            query.set('promoCodeName', promoCodeName);
        if (userAddress !== undefined)
            query.set('userAddress', userAddress);
        if (currency !== undefined)
            query.set('currency', currency);
        if (postalCode !== undefined)
            query.set('postalCode', postalCode);
        if (country !== undefined)
            query.set('country', country);
        return this.client.get("/checkout/price-by-id?".concat(query.toString()), this.headers());
    };
    /**
     * @description Fetches all available cryptocurrencies for the checkout module
     * @returns A promise which resolves to an array of available cryptocurrencies
     */
    Checkout.prototype.getAvailableCryptoCurrencies = function () {
        return this.client.get('/checkout/crypto/currencies', this.headers());
    };
    /**
     * @description Verifies a cryptocurrency payment
     * @param token - The encoded token we need to verify the payment
     * @returns A promise that resolves to a boolean indicating whether the payment is verified
     */
    Checkout.prototype.verifyCryptoPayment = function (token) {
        return this.client.post('/checkout/crypto/verify/payment', {
            token: token,
        }, this.authHeaders());
    };
    /**
     * Returns the needed headers with authorization header for the module requests
     * @private
     */
    Checkout.prototype.authHeaders = function () {
        return (0, headers_1.headersWithToken)({
            clientName: this.appDetails.clientName,
            clientVersion: this.appDetails.clientVersion,
            token: this.apiSecurity.token,
            workspaceToken: this.apiSecurity.workspaceToken,
            desktopToken: this.appDetails.desktopHeader,
        });
    };
    /**
     * Returns the basic needed headers for the module requests
     * @private
     */
    Checkout.prototype.headers = function () {
        var _a;
        var customHeaders = __assign({}, ((_a = this.appDetails.customHeaders) !== null && _a !== void 0 ? _a : {}));
        return (0, headers_1.basicHeaders)({
            clientName: this.appDetails.clientName,
            clientVersion: this.appDetails.clientVersion,
            customHeaders: customHeaders,
        });
    };
    return Checkout;
}());
exports.Checkout = Checkout;
