import { CreatedSubscriptionData } from '../drive/payments/types/types';
import { ApiSecurity, ApiUrl, AppDetails } from '../shared';
import { CreatePaymentIntentPayload, CreateSubscriptionPayload, CryptoCurrency, GetPriceByIdPayload, PaymentIntent, PriceWithTax } from './types';
export declare class Checkout {
    private readonly client;
    private readonly appDetails;
    private readonly apiSecurity;
    static client(apiUrl: ApiUrl, appDetails: AppDetails, apiSecurity: ApiSecurity): Checkout;
    private constructor();
    /**
     * @description Creates a customer or gets the existing one if it already exists
     * @param country - The country of the customer
     * @param postalCode - The postal code of the customer
     * @param companyVatId - The VAT ID of the company (optional)
     * @returns The customer ID and the user token used to create a subscription or payment intent
     */
    getCustomerId({ customerName, postalCode, country, captchaToken, companyVatId, }: {
        customerName: string;
        postalCode: string;
        country: string;
        captchaToken: string;
        companyVatId?: string;
    }): Promise<{
        customerId: string;
        token: string;
    }>;
    /**
     * @description Creates a subscription for a given customer
     * @param customerId - The ID of the customer
     * @param priceId - The ID of the price
     * @param token - The token used to authenticate the customer
     * @param currency - The currency of the subscription (optional)
     * @param promoCodeId - The ID of the promo code (optional)
     * @param quantity - The quantity of the subscription (optional)
     * @returns The created subscription data:
     * - `type`: The type of the subscription (setup or payment)
     * - `clientSecret`: The client secret for the subscription to be used with Stripe Elements
     * - `subscriptionId`: The ID of the subscription (optional)
     * - `paymentIntentId`: The ID of the payment intent (optional)
     */
    createSubscription({ customerId, priceId, token, currency, captchaToken, promoCodeId, quantity, }: CreateSubscriptionPayload): Promise<CreatedSubscriptionData>;
    /**
     * @description Creates a payment intent for a given customer
     * @param customerId - The ID of the customer
     * @param priceId - The ID of the price
     * @param token - The token used to authenticate the customer
     * @param currency - The currency of the payment intent (optional)
     * @param promoCodeId - The ID of the promo code (optional)
     * @returns The created invoice data:
     * - `clientSecret`: The client secret for the invoice to be used with Stripe Elements
     * - `id`: The ID of the invoice
     * - `invoiceStatus`: The status of the invoice (only when the status is 'paid')
     * - `type`: The type of the currency - 'fiat' or 'crypto'
     * - `payload`: The payload of the invoice if the type is 'crypto' containing { paymentRequestUri, url, qrUrl }
     * - `payload.paymentRequestUri`: The payment request URI for the invoice
     * - `payload.url`: The URL of the invoice
     * - `payload.qrUrl`: The QR code URL of the invoice
     */
    createPaymentIntent({ customerId, priceId, token, currency, captchaToken, promoCodeId, }: CreatePaymentIntentPayload): Promise<PaymentIntent>;
    /**
     * @description Fetch a requested price by its ID and its tax rate
     * @param priceId - The ID of the price
     * @param promoCodeName - The name of the promo code (optional)
     * @param currency - The currency of the price (optional)
     * @returns The price object containing the details of the requested price
     */
    getPriceById({ priceId, promoCodeName, userAddress, currency, postalCode, country, }: GetPriceByIdPayload): Promise<PriceWithTax>;
    /**
     * @description Fetches all available cryptocurrencies for the checkout module
     * @returns A promise which resolves to an array of available cryptocurrencies
     */
    getAvailableCryptoCurrencies(): Promise<CryptoCurrency[]>;
    /**
     * @description Verifies a cryptocurrency payment
     * @param token - The encoded token we need to verify the payment
     * @returns A promise that resolves to a boolean indicating whether the payment is verified
     */
    verifyCryptoPayment(token: string): Promise<boolean>;
    /**
     * Returns the needed headers with authorization header for the module requests
     * @private
     */
    private authHeaders;
    /**
     * Returns the basic needed headers for the module requests
     * @private
     */
    private headers;
}
