import { paths } from '../../schema';
import { ApiSecurity, ApiUrl, AppDetails } from '../../shared';
import { UserSettings } from '../../shared/types/userSettings';
import { ChangePasswordPayload, ChangePasswordPayloadNew, CheckChangeEmailExpirationResponse, FriendInvite, InitializeUserResponse, PreCreateUserResponse, Token, UpdateProfilePayload, UserPublicKeyResponse, UserPublicKeyWithCreationResponse, VerifyEmailChangeResponse } from './types';
export * as UserTypes from './types';
export declare class Users {
    private readonly client;
    private readonly appDetails;
    private readonly apiSecurity;
    static client(apiUrl: ApiUrl, appDetails: AppDetails, apiSecurity: ApiSecurity): Users;
    private constructor();
    /**
     * Sends an invitation to the specified email
     * @param email
     */
    sendInvitation(email: string): Promise<void>;
    /**
     * Initialize basic state of user and returns data after registration process
     * @param email
     * @param mnemonic
     */
    initialize(email: string, mnemonic: string): Promise<InitializeUserResponse>;
    /**
     * Returns fresh data of the user
     */
    refreshUser(): Promise<paths['/users/refresh']['get']['responses']['200']['content']['application/json']>;
    /**
     * Returns fresh avatar URL of the user
     */
    refreshAvatarUser(): Promise<{
        avatar: UserSettings['avatar'];
    }>;
    /**
     * Retrieves the user data for a specific user identified by the uuid.
     *
     * @param {string} params.userUuid - The UUID of the user.
     * @return {Promise<Object>} A promise that resolves to an object containing the user data.
     * The object has the following properties:
     * - `newToken` (string): The new token of the user.
     * - `oldToken` (string): The old drive token of the user.
     * - `user` (UserSettings): The user data.
     */
    getUserData({ userUuid }: {
        userUuid: string;
    }): Promise<{
        newToken: string;
        oldToken: string;
        user: UserSettings;
    }>;
    /**
     * Updates the authentication credentials and invalidates previous tokens (Legacy backend (drive-server))
     * @param payload
     *
     * @returns {Promise<{token: string, newToken: string}>} A promise that returns new tokens for this user.
     */
    changePasswordLegacy(payload: ChangePasswordPayload): Promise<{
        token: string;
        newToken: string;
    }>;
    /**
     * Updates the authentication credentials and invalidates previous tokens (New backend (drive-server-wip))
     * @param payload
     *
     * @returns {Promise<{token: string, newToken: string}>} A promise that returns new tokens for this user.
     */
    changePassword(payload: ChangePasswordPayloadNew): Promise<{
        token: string;
        newToken: string;
    }>;
    /**
     * Pre registers an email
     * @param email
     * @returns {Promise<PreCreateUserResponse>} A promise that returns a public key for this user.
     */
    preRegister(email: string): Promise<PreCreateUserResponse>;
    /**
     * @deprecated Use `updateUserProfile` instead.
     * Updates a user profile
     * @param payload
     */
    updateProfile(payload: UpdateProfilePayload): Promise<void>;
    /**
     * Updates a user profile
     * @param payload
     */
    updateUserProfile(payload: UpdateProfilePayload, token?: Token): Promise<void>;
    /**
     * @deprecated Use `updateUserAvatar` instead.
     * Updates a user avatar
     * @param payload
     */
    updateAvatar(payload: {
        avatar: Blob;
    }): Promise<{
        avatar: string;
    }>;
    /**
     * Updates a user avatar
     * @param payload
     */
    updateUserAvatar(payload: {
        avatar: Blob;
    }, token?: Token): Promise<{
        avatar: string;
    }>;
    /**
     * @deprecated Use `deleteUserAvatar` instead.
     * Delete current user avatar
     */
    deleteAvatar(): Promise<void>;
    /**
     * Delete current user avatar
     */
    deleteUserAvatar(token?: Token): Promise<void>;
    /**
     * Gets all friend invites created by this user
     */
    getFriendInvites(): Promise<FriendInvite[]>;
    /**
     * Sends verification email
     */
    sendVerificationEmail(token?: Token): Promise<void>;
    /**
     * Verifies user email
     */
    verifyEmail(payload: {
        verificationToken: string;
    }): Promise<void>;
    /**
     * Change user email by new email
     *
     * @param {string} newEmail
     *
     * @returns {Promise<void>}
     */
    changeUserEmail(newEmail: string): Promise<void>;
    /**
     * Verify user email change
     *
     * @param {string} encryptedAttemptChangeEmailId
     *
     * @returns {Promise<VerifyEmailChangeResponse>}
     */
    verifyEmailChange(encryptedAttemptChangeEmailId: string): Promise<VerifyEmailChangeResponse>;
    /**
     * Check if user email change verification link is expired
     *
     * @param {string} encryptedAttemptChangeEmailId
     *
     * @returns {Promise<CheckChangeEmailExpirationResponse>}
     */
    checkChangeEmailExpiration(encryptedAttemptChangeEmailId: string): Promise<CheckChangeEmailExpirationResponse>;
    /**
     * Get public key of given email
     */
    getPublicKeyByEmail({ email }: {
        email: string;
    }): Promise<UserPublicKeyResponse>;
    /**
     * Get public key of given email, if not exists it pre-create user with this email
     * and returns public key
     * @param email
     * @returns {Promise<UserPublicKeyWithCreationResponse>} A promise that returns the public keys of given user
     */
    getPublicKeyWithPrecreation({ email }: {
        email: string;
    }): Promise<UserPublicKeyWithCreationResponse>;
    /**
     * Generate mnemonic
     */
    generateMnemonic(): Promise<{
        mnemonic: string;
    }>;
    private basicHeaders;
    private headers;
    private headersWithToken;
}
