import { ApiSecurity, ApiUrl, AppDetails } from '../../shared';
import { AcceptInvitationToSharedFolderPayload, CreateSharingPayload, GenerateShareLinkPayload, GetShareLinkFolderSizePayload, GetSharedDirectoryPayload, ListAllSharedFoldersResponse, ListPrivateSharedFoldersResponse, ListShareLinksResponse, ListSharedItemsResponse, PublicSharedItemInfo, RemoveUserRolePayload, Role, ShareDomainsResponse, ShareFolderWithUserPayload, ShareLink, SharedFolderSize, SharedFolderUser, SharedFoldersInvitationsAsInvitedUserResponse, SharingInfo, SharingInvite, SharingMeta, UpdateShareLinkPayload, UpdateUserRolePayload, UpdateUserRoleResponse } from './types';
import { ItemType } from '../../workspaces';
export * as ShareTypes from './types';
export declare class Share {
    private readonly client;
    private readonly appDetails;
    private readonly apiSecurity;
    static client(apiUrl: ApiUrl, appDetails: AppDetails, apiSecurity: ApiSecurity): Share;
    private constructor();
    /**
     * Fetches the list of shared items
     */
    getShareLinks(page?: number, perPage?: number, orderBy?: 'views:ASC' | 'views:DESC' | 'createdAt:ASC' | 'createdAt:DESC'): Promise<ListShareLinksResponse>;
    /**
     * Creates a new link to share a file or folder
     * @param payload
     */
    createShareLink(payload: GenerateShareLinkPayload): Promise<{
        created: boolean;
        token: string;
        code: string;
    }>;
    /**
     * Update share link
     * @param payload
     */
    updateShareLink(payload: UpdateShareLinkPayload): Promise<ShareLink>;
    /**
     * Delete share link by id
     * @param payload
     */
    deleteShareLink(shareId: string): Promise<{
        deleted: boolean;
        shareId: string;
    }>;
    /**
     * Increment share view by token
     * @param token
     */
    incrementShareViewByToken(token: string): Promise<{
        incremented: boolean;
        token: string;
    }>;
    /**
     * Fetches data of a shared file
     * @param token
     */
    getShareLink(token: string, password?: string): Promise<ShareLink>;
    /**
     * Fetches paginated folders or files of a specific share link
     * @param payload
     */
    getShareLinkDirectory(payload: GetSharedDirectoryPayload): Promise<any>;
    getShareDomains(): Promise<ShareDomainsResponse>;
    /**
     * Get size of folder in share links
     * @param payload
     */
    getShareLinkFolderSize(payload: GetShareLinkFolderSizePayload): Promise<any>;
    /**
     * Fetches all folders shared by a user.
     *
     * @param {number} page - The page number for pagination.
     * @param {number} perPage - The number of items per page for pagination.
     * @param {string} [orderBy] - The optional order criteria (e.g., 'views:ASC', 'createdAt:DESC').
     * @returns {Promise<ListPrivateSharedFoldersResponse>} A promise containing the list of shared folders.
     */
    getSentSharedFolders(page?: number, perPage?: number, orderBy?: 'views:ASC' | 'views:DESC' | 'createdAt:ASC' | 'createdAt:DESC'): Promise<ListPrivateSharedFoldersResponse>;
    /**
     * Fetches folders shared with a user.
     *
     * @param {number} page - The page number for pagination.
     * @param {number} perPage - The number of items per page for pagination.
     * @param {string} [orderBy] - The optional order criteria (e.g., 'views:ASC', 'createdAt:DESC').
     * @returns {Promise<ListPrivateSharedFoldersResponse>} A promise containing the list of shared folders.
     */
    getReceivedSharedFolders(page?: number, perPage?: number, orderBy?: 'views:ASC' | 'views:DESC' | 'createdAt:ASC' | 'createdAt:DESC'): Promise<ListPrivateSharedFoldersResponse>;
    /**
     * Fetches all shared folders.
     *
     * @param {number} page - The page number for pagination.
     * @param {number} perPage - The number of items per page for pagination.
     * @param {string} [orderBy] - The optional order criteria (e.g., 'views:ASC', 'createdAt:DESC').
     * @returns {Promise<ListAllSharedFoldersResponse>} A promise containing the list of shared folders.
     */
    getAllSharedFolders(page?: number, perPage?: number, orderBy?: 'views:ASC' | 'views:DESC' | 'createdAt:ASC' | 'createdAt:DESC'): Promise<ListAllSharedFoldersResponse>;
    /**
     * Fetches all shared files.
     *
     * @param {number} page - The page number for pagination.
     * @param {number} perPage - The number of items per page for pagination.
     * @param {string} [orderBy] - The optional order criteria (e.g., 'views:ASC', 'createdAt:DESC').
     * @returns {Promise<ListAllSharedFoldersResponse>} A promise containing the list of shared folders.
     */
    getAllSharedFiles(page?: number, perPage?: number, orderBy?: 'views:ASC' | 'views:DESC' | 'createdAt:ASC' | 'createdAt:DESC'): Promise<ListAllSharedFoldersResponse>;
    /**
     * Get all users with access to a shared folder.
     *
     * @param {string} folderUUID - The UUID of the folder.
     * @param {number} page - The page number for pagination.
     * @param {number} perPage - The number of items per page for pagination.
     * @param {string} [orderBy] - The optional order criteria (e.g., 'views:ASC', 'createdAt:DESC').
     * @returns {Promise<{ users: SharedFolderUser[] }>} A promise containing the list of users with access to the folder.
     */
    getSharedFolderUsers(folderUUID: string, page?: number, perPage?: number, orderBy?: 'views:ASC' | 'views:DESC' | 'createdAt:ASC' | 'createdAt:DESC'): Promise<{
        users: SharedFolderUser[];
    }>;
    /**
     * Get shared folder content
     * @param {string} sharedFolderId - The UUID of the shared folder.
     * @param {string} type - The item type for the query folders/files
     * @param {string} token - Key that enables invited users to navigate the folders
     * @param {number} page - The page number for pagination.
     * @param {number} perPage - The number of items per page for pagination.
     * @param {string} [orderBy] - The optional order criteria (e.g., 'views:ASC', 'createdAt:DESC').
     */
    getSharedFolderContent(sharedFolderId: string, type: 'folders' | 'files', token: string | null, page?: number, perPage?: number, orderBy?: 'views:ASC' | 'views:DESC' | 'createdAt:ASC' | 'createdAt:DESC'): Promise<ListSharedItemsResponse>;
    /**
     * Get public shared folder content
     * @param {string} sharedFolderId - The UUID of the shared folder.
     * @param {string} type - The item type for the query folders/files
     * @param {string} token - Key that enables invited users to navigate the folders
     * @param {number} page - The page number for pagination.
     * @param {number} perPage - The number of items per page for pagination.
     * @param {string} [orderBy] - The optional order criteria (e.g., 'views:ASC', 'createdAt:DESC').
     */
    getPublicSharedFolderContent(sharedFolderId: string, type: 'folders' | 'files', token: string | null, page?: number, perPage?: number, code?: string, orderBy?: 'views:ASC' | 'views:DESC' | 'createdAt:ASC' | 'createdAt:DESC'): Promise<ListSharedItemsResponse>;
    /**
     * Get the role of a user on a folder.
     *
     * @param {string} options.sharingId - The unique identifier of the sharing.
     * @returns {Promise<Role>} A promise containing the role of the current user in the sharing.
     */
    getUserRole(sharingId: string): Promise<Role>;
    /**
     * Update the role of a user on a folder.
     *
     * @param {UpdateUserRolePayload} options - The options for updating the user's role on the folder.
     * @param {string} options.sharingId - The unique identifier of the user to whom we will update the role.
     * @param {string} options.newRoleId - The new role Id.
     * @returns {Promise<UpdateRoleFolderResponse>} A promise that resolves when the user's role is updated.
     */
    updateUserRole({ sharingId, newRoleId }: UpdateUserRolePayload): Promise<UpdateUserRoleResponse>;
    /**
     * Remove user from shared folder.
     *
     * @param {UpdateUserRolePayload} options - The options for updating the user's role on the folder.
     * @param {string} options.itemType - The unique identifier of the folder.
     * @param {string} options.itemId - The identifier of the role to assign to the user.
     * @param {string} options.userId - The role Id how we want to delete.
     * @returns {Promise<UpdateRoleFolderResponse>} A promise that resolves when the user's role is updated.
     */
    removeUserRole({ itemType, itemId, userId }: RemoveUserRolePayload): Promise<UpdateUserRoleResponse>;
    /**
     * Get private folder data.
     *
     * @param {string} itemId - The itemId of the folder.
     * @param {string} itemType - The itemType of the folder (file | folder).
     * @returns {Promise<{ data: SharingInvite[] }>} A promise containing the private folder data.
     */
    getSharedFolderInvitations({ itemId, itemType }: {
        itemId: string;
        itemType: ItemType;
    }): Promise<any[]>;
    /**
     * Get all invitations for a user.
     * @param limit - The number of items per page for pagination.
     * @param offset - The page number for pagination.
     * @returns {Promise<invites: any>} A promise containing the list of invitations.
     */
    getSharedFolderInvitationsAsInvitedUser({ limit, offset, }: {
        limit?: number;
        offset?: number;
    }): Promise<{
        invites: SharedFoldersInvitationsAsInvitedUserResponse[];
    }>;
    /**
     * Share a private folder with a user.
     *
     * @param {ShareFolderWithUserPayload} options - The options for sharing the private folder with a user.
     * @param {string} options.itemId - The id of the item to share.
     * @param {string} options.itemType - The type of the item to share (folder | file).
     * @param {string} options.sharedWith - The email address of the user to share the folder with.
     * @param {string} options.encryptionKey - Owner\'s encryption key encrypted with the invited user\'s public key. This field should not be empty if the invitation type is "OWNER".
     * @param {string} options.encryptionAlgorithm - Encryption algorithm used to encrypt the encryption key. This field should not be empty if the invitation type is "OWNER".
     * @param {string} options.type - Owner's encryption key encrypted with the invited user's public key.
     * @param {string} options.roleId - The id of the role to assign to the user.
     * @param {string} options.notifyUser - If it has to notify the users
     * @param {string} options.notificationMessage - Message of the notificacion
     *
     *
     * @returns {Promise<SharingInvite>} A promise that resolves when the folder is shared with the user.
     */
    inviteUserToSharedFolder(createInviteDto: ShareFolderWithUserPayload): Promise<SharingInvite>;
    /**
     * Create a sharing.
     */
    createSharing(createSharingPayload: CreateSharingPayload): Promise<SharingMeta>;
    /**
     * Get sharing meta with code
     */
    getSharingMeta(sharingId: string, code: string, password?: string): Promise<SharingMeta>;
    /**
     * Add/edit sharing Password
     * @param {string} sharingId - id of sharing.
     * @param {string} encryptedPassword - password encrypted with CODE as key
     * @returns {Promise<SharingMeta>} A promise that returns the sharing info with the new encrypted password
     */
    saveSharingPassword(sharingId: string, encryptedPassword: string): Promise<SharingMeta>;
    /**
     * Remove password protection from sharing
     * @param {string} sharingId - id of sharing.
     * @returns {Promise<void>} A promise that resolves when password was successfully deleted.
     */
    removeSharingPassword(sharingId: string): Promise<void>;
    /**
     * Get public information of the item shared.
     * @param {string} sharingId - id of sharing.
     * @returns {Promise<PublicSharedItemInfo>} A promise that returns data of the public shared item.
     */
    getPublicSharedItemInfo(sharingId: string): Promise<PublicSharedItemInfo>;
    /**
     * Request access to shared folder.
     */
    requestUserToSharedFolder(createInviteDto: ShareFolderWithUserPayload): Promise<void>;
    /**
     * Check if the expirationDate of invite is valid.
     * @param {string} invitationId - The id of the invitation.
     
     * @returns {Promise<{uuid: string}>}  A promise returning the uuid of the invitation if valid.
     */
    validateInviteExpiration(invitationId: string): Promise<{
        uuid: string;
    }>;
    /**
     * Share a private folder with a user.
     * @param {string} invitationId - The id of the invitation.
     * @param {ShareFolderWithUserPayload} options - The options for sharing the private folder with a user.
     * @param {string} options.encryptionKey - The encryption key (just in case the invitation is a request).
     * @param {string} options.itemType - The encryption algorithm (just in case the invitation is a request).
     
     * @returns {Promise<void>} A promise that resolves when the folder is shared with the user.
     */
    acceptSharedFolderInvite({ invitationId, acceptInvite, token, }: {
        invitationId: string;
        acceptInvite?: AcceptInvitationToSharedFolderPayload;
        token?: string;
    }): Promise<void>;
    /**
     * Change Sharing Mode.
     * @param {string} options.itemType - folder | file
     * @param {string} options.itemId - id of folder or file
     * @param {string} options.sharingType - New Sharing type.
     
     * @returns {Promise<void>} A promise that resolves when sharing mode has been updated.
     */
    updateSharingType({ itemId, itemType, sharingType, }: {
        itemId: string;
        itemType: string;
        sharingType: string;
    }): Promise<void>;
    /**
     * Get Sharing type
     * @param {string} options.itemType - folder | file
     * @param {string} options.itemId - id of folder or file
     * @returns {Promise<SharingMeta>} A promise that returns the sharing data.
     */
    getSharingType({ itemId, itemType }: {
        itemId: string;
        itemType: string;
    }): Promise<SharingMeta>;
    /**
     * Obtains detailed sharing information for a given file or folder
     * @param {string} options.itemType - folder | file
     * @param {string} options.itemId - id of folder or file
     * @returns {Promise<SharingInfo>} A promise that returns information about the shared item.
     */
    getSharingInfo({ itemId, itemType }: {
        itemId: string;
        itemType: string;
    }): Promise<SharingInfo>;
    declineSharedFolderInvite(invitationId: string, token?: string): Promise<void>;
    /**
     * Fetches roles for sharing items.
     *
     * @returns {Promise<PrivateSharingRolesResponse>} A promise containing the list of sharing roles.
     */
    getSharingRoles(): Promise<Role[]>;
    getAllAccessUsers({ itemType, folderId, }: {
        itemType: string;
        folderId: string;
    }): Promise<Record<'users', any[]> | Record<'error', string>>;
    /**
     * Stop sharing folder
     * @param {string} itemType - Type of the sharing to delete
     * @param {string} itemId - Id of the sharing to delete
     * @returns
     */
    stopSharingFolder(itemType: string, itemId: string): Promise<void>;
    /**
     * Returns the needed headers for the module requests
     * @private
     */
    private headers;
    /**
     * Returns the needed headers for the module requests
     * @private
     */
    private basicHeaders;
    /**
     * Used to send the password in shares
     * @private
     */
    private basicHeadersWithPassword;
    /**
     * Get request headers with optional authorization token.
     *
     * @param {string} [token] - Optional authorization token.
     * @returns {Object} - Request headers object.
     */
    private getRequestHeaders;
    /**
     * Gets the size of a shared folder given sharing id
     *
     * @param {string} sharingId - Sharing ID.
     * @returns {Promise<SharedFolderSize>}
     */
    getSharedFolderSize(id: string): Promise<SharedFolderSize>;
}
