"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Backups = void 0;
var client_1 = require("../../shared/http/client");
var headers_1 = require("../../shared/headers");
var Backups = /** @class */ (function () {
    function Backups(apiUrl, appDetails, apiSecurity) {
        this.client = client_1.HttpClient.create(apiUrl, apiSecurity.unauthorizedCallback);
        this.appDetails = appDetails;
        this.apiSecurity = apiSecurity;
    }
    Backups.client = function (apiUrl, appDetails, apiSecurity) {
        return new Backups(apiUrl, appDetails, apiSecurity);
    };
    /**
     * @deprecated Use `getBackupDevices` instead.
     * This method uses the old drive backend, while `getBackupDevices` uses the new drive backend.
     */
    Backups.prototype.getAllDevices = function () {
        return this.client.get('/backup/device', this.headers());
    };
    /**
     * Retrieves the list of backup devices associated with the user's account.
     *
     * @returns {Promise<Device[]>} A promise that resolves to an array of Devices.
     */
    Backups.prototype.getBackupDevices = function () {
        return this.client.get('/backup/devices', this.headers());
    };
    /**
     * Retrieves a list of all devices represented as folders.
     *
     * This method sends a GET request to the `/backup/deviceAsFolder` endpoint
     * and returns an array of `DriveFolderData` objects, each representing a device
     * as a folder in the drive.
     *
     * @returns {Promise<DriveFolderData[]>} A promise that resolves to an array of DriveFolderData.
     */
    Backups.prototype.getAllDevicesAsFolder = function () {
        return this.client.get('/backup/deviceAsFolder', this.headers());
    };
    /**
     * Retrieves all backups associated with a specific device identified by its mac ID.
     *
     * @param mac - The mac ID of the device for which backups are to be retrieved.
     * @returns A promise that resolves to an array of DeviceBackups.
     */
    Backups.prototype.getAllBackups = function (mac) {
        return this.client.get("/backup/".concat(mac), this.headers());
    };
    /**
     * Deletes a backup by its ID.
     *
     * @param backupId - The unique identifier of the backup to be deleted.
     * @returns A promise that resolves when the backup is successfully deleted.
     */
    Backups.prototype.deleteBackup = function (backupId) {
        return this.client.delete("/backup/".concat(backupId), this.headers());
    };
    /**
     * @deprecated Use `deleteBackupDevice` instead.
     * This method uses the old drive backend, while `deleteBackupDevice` uses the new drive backend.
     */
    Backups.prototype.deleteDevice = function (deviceId) {
        return this.client.delete("/backup/device/".concat(deviceId), this.headers());
    };
    /**
     * Deletes a backup device by its ID.
     *
     * @param deviceId - The unique identifier of the device to be deleted.
     * @returns A promise that resolves when the device is successfully deleted.
     */
    Backups.prototype.deleteBackupDevice = function (deviceId) {
        return this.client.delete("/backup/devices/".concat(deviceId), this.headers());
    };
    /**
     * Deletes a backup device by its folder ID.
     *
     * @param folderId - The unique identifier of the folder to be deleted.
     * @returns A promise that resolves when the device is successfully deleted.
     */
    Backups.prototype.deleteBackupDeviceAsFolder = function (folderId) {
        return this.client.delete("/backup/deviceAsFolder/".concat(folderId), this.headers());
    };
    /**
     * Returns the needed headers for the module requests
     * @private
     */
    Backups.prototype.headers = function () {
        return (0, headers_1.headersWithToken)({
            clientName: this.appDetails.clientName,
            clientVersion: this.appDetails.clientVersion,
            token: this.apiSecurity.token,
            workspaceToken: this.apiSecurity.workspaceToken,
            desktopToken: this.appDetails.desktopHeader,
            customHeaders: this.appDetails.customHeaders,
        });
    };
    return Backups;
}());
exports.Backups = Backups;
